<?php

namespace MagnewsAdaptater;

use Exception;
use MagnewsAdaptater\Error\MagnewsException;
use MagnewsAdaptater\MagnewsDAO\Credentials;
use MagnewsAdaptater\MagnewsDAO\MagnewsSoap;
use MagnewsAdaptater\Mailer\MagnewsMailer;
use Monolog\Logger;

/**
 * Singleton pour interagir avec l'API Magnews
 */
class MagnewsAdaptater
{

    /**
     * SOAP credentials
     **/
    private Credentials $credentials;
    private static MagnewsAdaptater|null $_instance;
    private Logger $logger;
    private string $app_path;

    /**
     * Permet de récupérer l'instance
     * @throws MagnewsException
     */
    public static function getInstance(Logger $logger = null, string $app_path = null): MagnewsAdaptater
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new MagnewsAdaptater($logger, $app_path);
        }
        return self::$_instance;
    }

    public function getLogger(): Logger
    {
        return $this->logger;
    }

    /**
     * @return Credentials
     */
    public function getCredentials(): Credentials
    {
        return $this->credentials;
    }

    /**
     * @throws \SoapFault|MagnewsException
     */
    public function getSoapClient($wsdl_url): MagnewsSoap
    {
        return new MagnewsSoap($wsdl_url);
    }

    /**
     * @throws MagnewsException
     */
    private function __construct(Logger $logger, string $app_path)
    {
        $this->logger = $logger;
        $this->app_path = $app_path;

        try {
            $credentials = new Credentials();
            $credentials->password = file_get_contents($this->app_path . '/mag.key');
            $this->credentials = $credentials;
        } catch (Exception $exception) {
            $this->logger->error(
                self::class . ' : Erreur à la récupération de la clé Magnews',
                [
                    'message' => "'" . $this->app_path . '/mag.key' . "'" . ' n\'a pas pu être récupéré, le fichier existe ?',
                    'exception' => $exception
                ]
            );
            throw new MagnewsException($exception->getMessage(), "Erreur à la récupération de la clé Magnews");
        }
    }

    public function sendTransactionalMail($fromEmail, $fromName, $replyTo, $to, $subject, $idTemplate, $typedValues = [], $idMessage = ''): bool
    {
        try {
            return MagnewsMailer::sendTransactionalMailMagnews($this->credentials, $fromEmail, $fromName, $replyTo, $to, $subject, $idTemplate, $typedValues, $idMessage);
        } catch (MagnewsException|Exception $exception) {
            $this->logger->error(__FUNCTION__ . "Erreur à l'envoi du mail par Magnews", ['exception' => $exception]);
            return false;
        }
    }

    public static function tearDown() {
        static::$_instance = null;
    }
}