<?php
/**
 * Created by IntelliJ IDEA.
 * User: Loïc Rameau
 * Date: 16/03/2017
 * Time: 14:17
 */

namespace Bloom\LaravelSendinblue;

use Exception;
use GuzzleHttp\Client;
use Illuminate\Mail\Transport\Transport;
use Illuminate\Support\Facades\Log;
use SendinBlue\Client\Api\SMTPApi;
use SendinBlue\Client\Configuration;
use SendinBlue\Client\Model\SendSmtpEmail;
use SendinBlue\Client\Model\SendSmtpEmailAttachment;
use SendinBlue\Client\Model\SendSmtpEmailBcc;
use SendinBlue\Client\Model\SendSmtpEmailCc;
use SendinBlue\Client\Model\SendSmtpEmailReplyTo;
use SendinBlue\Client\Model\SendSmtpEmailSender;
use SendinBlue\Client\Model\SendSmtpEmailTo;
use Swift_Attachment;
use Swift_MimePart;

class SendinBlueTransport extends Transport
{
    /**
     * @var Configuration
     */
    protected $config;
    /**
     * @var string
     */
    protected $tag;
    /**
     * @var object
     */
    protected $apiInstance;

    public function __construct(Configuration $config, string $tag)
    {
        $this->tag = $tag;
        $this->apiInstance = new SMTPApi(
            new Client(),
            $config
        );

    }

    public function setTag($tag)
    {
        $this->tag = $tag;
    }

    /**
     * {@inheritdoc}
     * @throws SendinBlueException
     */
    public function send(\Swift_Mime_SimpleMessage $message, &$failedRecipients = null)
    {
        $this->beforeSendPerformed($message);


        $sendSmtpEmail = new SendSmtpEmail($this->buildData($message));

        try {
            $result = $this->apiInstance->sendTransacEmail($sendSmtpEmail);
            if ($result->valid() == false) {
                Log::error('Retour de mail non valide');
                throw new SendinBlueException('Retour de mail non valide');
            }


        } catch (Exception $e) {
            Log::debug("Exception sur l'envoi de mail: ", [$e->getMessage(),$e]);
            throw new SendinBlueException($e->getMessage(), $e->getCode(), $e);
        }


        // Should return the number of recipients who were accepted for delivery.
        return 0;

    }

    /**
     * Transforms Swift_Message into data array for SendinBlue's API
     * cf. https://apidocs.sendinblue.com/tutorial-sending-transactional-email/ (api v2)
     * cf. https://github.com/sendinblue/APIv3-php-library/blob/master/docs/Model/SendSmtpEmail.md (api v3)
     *
     * @param \Swift_Mime_SimpleMessage $message
     *
     * @return array
     */
    protected function buildData($message)
    {
        $data = [];
        if ($message->getTo()) {
            $data['to'] = [];

            foreach ($message->getTo() as $email => $name) {
                $to = new SendSmtpEmailTo([
                    'email' => $email,
                    'name'  => $name
                ]);
                $data['to'][] = $to;
            }
        }
        if ($message->getSubject()) {
            $data['subject'] = $message->getSubject();
        }
        if ($message->getFrom()) {
            $from = $message->getFrom();
            reset($from);
            $key = key($from);
            $data['sender'] = new SendSmtpEmailSender(['email' => $key, 'name' => $from[ $key ]]);
        }
        // set content
        if ($message->getContentType() == 'text/plain') {
            $data['textContent'] = $message->getBody();
        } else {
            $data['htmlContent'] = $message->getBody();
        }
        $children = $message->getChildren();
        foreach ($children as $child) {
            if ($child instanceof Swift_MimePart && $child->getContentType() == 'text/plain') {
                $data['textContent'] = $child->getBody();
            }
        }
        if (!isset($data['textContent'])) {
            $data['textContent'] = strip_tags($message->getBody());
        }
        // end set content
        if ($message->getCc()) {
            $data['cc'] = [];
            foreach ($message->getCc() as $email => $name) {
                $cc = new SendSmtpEmailCc([
                    'email' => $email,
                    'name'  => $email
                ]);
                $data['cc'][] = $cc;
            }
        }
        if ($message->getBcc()) {
            $data['bcc'] = [];
            foreach ($message->getBcc() as $email => $name) {
                $bcc = new SendSmtpEmailBcc([
                    'email' => $email,
                    'name'  => $name
                ]);
                $data['bcc'][] = $bcc;
            }
        }
        if ($message->getReplyTo()) {
            $replyTo = $message->getReplyTo();
            $adress = array_keys($replyTo)[0];
            $name = $replyTo[$adress];
            $data['replyTo'] = new SendSmtpEmailReplyTo(['email' => $adress, 'name' => $name]);
        }
        // attachment
        $attachment = [];
        foreach ($children as $child) {
            if ($child instanceof Swift_Attachment) {
                $filename = $child->getFilename();
                $content = chunk_split(base64_encode($child->getBody()));
                $attachment[] = new SendSmtpEmailAttachment([
                    'content' => $content,
                    'name'    => $filename
                ]);
            }
        }
        if (count($attachment)) {
            $data['attachment'] = $attachment;
        }
        $data['headers'] = [
            'X-Mailin-Tag' => $this->tag
        ];

        return $data;
    }
}
