<?php

namespace Tests\Feature;

use Bloom\LaravelMagnews\Facades\MagnewsApi;
use Tests\TestCase;

class MagnewsTest extends TestCase
{
    private int $idDatabaseMagnews;
    private string $contactNomeUtente;
    private string $contactEmail;
    private string $contactIdContact;
    private string $noResultMessage;
    private string $nomeUtenteNotExistsMessage;

    protected function setUp(): void
    {
        parent::setUp();
        $this->idDatabaseMagnews = env('ID_DATABASE_MAGNEWS',26); // bloom test
        $this->contactNomeUtente = env('CONTACT_NOME_UTENTE','m43al12qwq');
        $this->contactEmail = env('CONTACT_EMAIL','crouylou@bloom-multimedia.fr');
        $this->contactIdContact = env('CONTACT_ID_CONTACT','4209883');
        $this->noResultMessage = env('NO_RESULT_MESSAGE','Aucun résultat.');
        $this->nomeUtenteNotExistsMessage = env('NOME_UTENTE_NOT_EXISTS_MESSAGE',"Ce Nome Utente n'existe pas dans cette base.");
        $this->idSpecialist = env('ID_SPECIAL_LIST',13); // BLOOM_TEST_LISTE_SPECIALE
    }

    /**
     * test la sauvegarde d'un contact dans magnews
     */
    public function test_save_contact(): void
    {
        $response = MagnewsApi::saveContact(['EMAIL'=> $this->contactEmail], $this->idDatabaseMagnews);
        $this->assertTrue($response->ok);
    }
    /**
     * test le désabonnement d'un contact dans magnews
     */
    public function test_unsubscribe_contact_ok(): void
    {
        $response = MagnewsApi::unsubscribeContact($this->contactNomeUtente, $this->idDatabaseMagnews);
        $this->assertTrue($response->ok);
    }
    /**
     * test le désabonnement d'un contact dans magnews-ko
     */
    public function test_unsubscribe_contact_ko(): void
    {
        $response = MagnewsApi::unsubscribeContact('123', $this->idDatabaseMagnews);
        $this->assertEquals($this->nomeUtenteNotExistsMessage, $response->getData()->message);
    }

    /**
     * test l'abonnement d'un contact dans magnews
     */
    public function test_subscribe_contact_ok(): void
    {
        $response = MagnewsApi::subscribeContact($this->contactNomeUtente, $this->idDatabaseMagnews);
        $this->assertTrue($response->ok);
    }
    /**
     * test l'abonnement d'un contact dans magnews-ko
     */
    public function test_subscribe_contact_ko(): void
    {
        $response = MagnewsApi::subscribeContact('123', $this->idDatabaseMagnews);
        $this->assertEquals($this->nomeUtenteNotExistsMessage, $response->getData()->message);
    }
    /**
     * test récupère contact via nome utente -ok
     */
    public function test_get_contact_from_nome_utente_ok(): void
    {
        $response = MagnewsApi::getContactFromPrimaryKey($this->contactNomeUtente, $this->idDatabaseMagnews);
        $this->assertNotEmpty($response->email);
    }
    /**
     * test récupère contact via nome utente -ko
     */
    public function test_get_contact_from_nome_utente_ko(): void
    {
        $response = MagnewsApi::getContactFromPrimaryKey('123', $this->idDatabaseMagnews);
        $this->assertEquals($this->noResultMessage, $response->getData()->message);
    }
    /**
     * test récupère contact via id contact - ok
     */
    public function test_get_contact_from_id_contact_ok(): void
    {
        $response = MagnewsApi::getContactFromIdContact($this->contactIdContact);
        $this->assertNotEmpty($response->email);
    }
    /**
     * test récupère contact via id contact - KO
     */
    public function test_get_contact_from_id_contact_ko(): void
    {
        $response = MagnewsApi::getContactFromIdContact('123');
        $this->assertEquals($this->noResultMessage, $response->getData()->message);
    }
    /**
     * test requête sur les contacts - OK
     */
    public function test_query_contacts_all_fields_ok(): void
    {
        $response = MagnewsApi::queryContactsAllFields("email ='$this->contactEmail'", $this->idDatabaseMagnews);

        if(is_array($response)){
            $this->assertNotEmpty($response[0]->email);
        }
        else{
            $this->assertNotEmpty($response->email);
        }
    }
    /**
     * test requête sur les contacts - KO
     */
    public function test_query_contacts_all_fields_ko(): void
    {
        $response = MagnewsApi::queryContactsAllFields("email ='emailExistePas'", $this->idDatabaseMagnews);
        $this->assertEquals($this->noResultMessage, $response->getData()->message);

    }
    /**
     * test requête count sur les contacts - OK
     */
    public function test_query_count_contacts_ok(): void
    {
        $response = MagnewsApi::queryCountContacts("email ='$this->contactEmail'", $this->idDatabaseMagnews);
        $this->assertGreaterThan(0,$response);
    }
    /**
     * test requête count sur les contacts - aucun contact
     */
    public function test_query_count_0_contacts(): void
    {
        $response = MagnewsApi::queryCountContacts("email ='emailExistePas'", $this->idDatabaseMagnews);
        $this->assertEquals(0,$response);

    }
    /**
     * test récupérer les contacts qui appartiennent à une liste spéciale- OK
     */
    public function test_query_contacts_belongs_to_special_list_ok(): void
    {
        $response = MagnewsApi::queryContactsBelongsToSpecialList($this->idSpecialist, $this->idDatabaseMagnews);
        if(is_array($response)){
            $this->assertNotEmpty($response[0]->email);
        }
        else{
            $this->assertNotEmpty($response->email);
        }
    }
    /**
     * test récupérer les contacts qui appartiennent à une liste spéciale- KO
     */
    public function test_query_contacts_belongs_to_special_list_ko(): void
    {
        $this->expectException(\Exception::class);
        $response = MagnewsApi::queryContactsBelongsToSpecialList(123, $this->idDatabaseMagnews);
        $this->assertThrows($response,\Exception::class);
    }
    /**
     * test Returns the definition for a field in a DB contacts by ID.- OK
     */
    public function test_get_BDD_field_by_id_ok(): void
    {
        $response = MagnewsApi::getBDDFieldById("BLOOM_TEST_AUDIENCE", $this->idDatabaseMagnews);
        $this->assertNotEmpty($response->name);
    }
    /**
     * test Returns the definition for a field in a DB contacts by ID.- KO
     */
    public function test_get_BDD_field_by_id_ko(): void
    {
        $response = MagnewsApi::getBDDFieldById("CHAMP_EXISTE_PAS", $this->idDatabaseMagnews);
        $this->assertEquals($this->noResultMessage, $response->getData()->message);
    }
    /**
     * test Returns basic information about all Databases.- OK
     */
    public function test_get_BDDs_basic_infos_ok(): void
    {
        $response = MagnewsApi::getBDDsBasicInfos();
        $this->assertGreaterThan(0,$response);
    }
    /**
     * test Returns basic information about a Database.- OK
     */
    public function test_get_BDD_basic_infos_by_id_ok(): void
    {
        $response = MagnewsApi::getBDDBasicInfosById($this->idDatabaseMagnews);
        $this->assertNotEmpty($response->idDatabase);
    }
    /**
     * test Returns basic information about a Database.- KO
     */
    public function test_get_BDD_basic_infos_by_id_ko(): void
    {
        $response = MagnewsApi::getBDDBasicInfosById('123');
        $this->assertEquals($this->noResultMessage, $response->getData()->message);
    }
    /**
     * test Returns the definition of all fields for a Database.- OK
     */
    public function test_get_BDD_fields_ok(): void
    {
        $response = MagnewsApi::getBDDFields($this->idDatabaseMagnews);
        $this->assertGreaterThan(0,$response);
    }
    /**
     * test Returns the definition of all fields for a Database.- KO
     */
    public function test_get_BDD_fields_ko(): void
    {
        $this->expectException(\Exception::class);
        $response = MagnewsApi::getBDDFields('123');
        $this->assertThrows($response, \Exception::class);
    }
    /**
     * test  Execute a workflow with only id contact variable - OK
     */
    public function test_enter_workflow_id_contact_ok(): void
    {
        $response = MagnewsApi::enterWorkflowWithIdContact($this->contactIdContact,'API_TEST');
        $this->assertTrue($response);
    }
    /**
     * test  Execute a workflow with only id contact variable - KO
     */
    public function test_enter_workflow_id_contact_ko(): void
    {
        $response = MagnewsApi::enterWorkflowWithIdContact(123,'API_TEST');
        $this->assertFalse($response);
    }
    /**
     * test  Execute a workflow - OK
     */
    public function test_enter_workflow_ok(): void
    {
        $response = MagnewsApi::enterWorkflow([
            "idContact" => $this->contactIdContact
        ],'API_TEST');
        $this->assertTrue($response);
    }
    /**
     * test  Execute a workflow - KO
     */
    public function test_enter_workflow_ko(): void
    {
        $response = MagnewsApi::enterWorkflow([
            "idContact" => 123
        ],'API_TEST');
        $this->assertFalse($response);
    }

    /**
     * test  get form submissions results - empty
     */
    public function test_get_form_submissions_results_empty(): void
    {
        $response = MagnewsApi::getFormSubmissionsResults(5);
        $this->assertEquals($this->noResultMessage, $response->getData()->message);

    }
    //TODO test  : faire un test avec un formulaire rempli
    /**
     * test  get survey sessions results - que les résultats complétés
     */
    public function test_get_survey_results_completed(): void
    {
        $response = MagnewsApi::getSurveyResults(11, null,null,true, false);
        $this->assertNotEmpty($response);
    }
    /**
     * test  get all fields in survey
     */
    public function test_get_all_fields_in_survey(): void
    {
        $response = MagnewsApi::getSurveyFields(11);
        $this->assertNotEmpty($response);
    }

}

