<?php

namespace Bloom\LaravelMagnews\Services;

use Bloom\LaravelMagnews\DOT\Credentials;
use Bloom\LaravelMagnews\Helpers\MagnewsHelper;
use DateTime;
use Exception;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class MagnewsService
{
    protected MagnewsSoap $magnewsSoap;
    protected Credentials $credentials;

    /**
     * @throws \SoapFault
     */
    public function __construct()
    {
        $this->magnewsSoap = new MagnewsSoap(config('magnews.url'));
        $credentials = new Credentials();
        $credentials->password = config('magnews.key');
        $this->magnewsSoap->setCredentials($credentials);
    }

    /**
     * Fonction permettant d'enregistrer un contact dans Magnews
     * @param array $contact
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function saveContact(Array $contact, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->mergeContact([
                'idDatabase' => $idDatabase,
                'values'     => MagnewsHelper::formatContactValues($contact),
            ]);

            Log::info(
                __METHOD__ . ' : Envoi vers Magnews du contact réussi ' . json_encode($response->return). ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Envoi vers Magnews du contact échoué ' . json_encode($contact). ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Envoi vers Magnews du contact échoué ' . json_encode($contact). ' dans la BDD ' . $idDatabase . '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de désabonner un contact dans Magnews
     * @param string $nomeUtente
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function unsubscribeContact(string $nomeUtente, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->unsubscribeContact([
                'idDatabase' => $idDatabase,
                'values'     => MagnewsHelper::createContactValues('NOME_UTENTE',$nomeUtente),
            ]);

            if($response->return->actionResult == "insert"){
                //le contact n'existe pas. La fonction  subscribeContact essaie de l'insérer
                return $this->nomeUtenteNotExists();
            }

            //success
            Log::info(
                __METHOD__ . ' : Désabonnement du contact réussi ' . json_encode($response->return). ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Désabonnement du contact échoué. Nome Utente: ' . $nomeUtente. ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Désabonnement du contact échoué. Nome Utente: ' . $nomeUtente. ' dans la BDD ' . $idDatabase . '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant d'abonner un contact dans Magnews
     * @param string $nomeUtente
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function subscribeContact(string $nomeUtente, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->subscribeContact([
                'idDatabase' => $idDatabase,
                'values'     => MagnewsHelper::createContactValues('NOME_UTENTE',$nomeUtente),
            ]);

            if($response->return->actionResult == "insert"){
                //le contact n'existe pas. La fonction  subscribeContact essaie de l'insérer
                return $this->nomeUtenteNotExists();
            }

            //success
            Log::info(
                __METHOD__ . ' : Abonnement du contact réussi ' . json_encode($response->return). ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Abonnement du contact échoué. Nome Utente: ' . $nomeUtente. ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Abonnement du contact échoué. Nome Utente: ' . $nomeUtente. ' dans la BDD ' . $idDatabase . '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer un contact dans Magnews via clé primaire
     * @param string $primaryKey
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getContactFromPrimaryKey(string $primaryKey, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findContactByPrimaryKey([
                'idDatabase' => $idDatabase,
                'primaryKeyValue' => $primaryKey,
            ]);

            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération du contact échoué. Primary key:' . $primaryKey. ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération du contact échoué. Primary key: ' .$primaryKey. ' dans la BDD ' . $idDatabase . '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer un contact dans Magnews via id contact
     * @param string $idContact
     * @return mixed
     * @throws Exception
     */
    public function getContactFromIdContact(string $idContact): mixed
    {
        try {
            $response = $this->magnewsSoap->findContactById([
                'idContact' => $idContact,
            ]);

            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération du contact échoué. idContact: '.$idContact,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération du contact échoué. idContact: '.$idContact.' Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de faire une requête  sur les contacts en récupérant tous les champs
     * @param string $where
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryContactsAllFields(string $where, int $idDatabase): mixed
    {
        try {
            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->queryContacts([
                'idDatabase' => $idDatabase,
                'query' => MagnewsHelper::makeQuery($where),
            ]);

            //va chercher les infos des contacts
            $responseInfo = $this->magnewsSoap->fetchContacts([
                'idRowSet'=>$response->return,
                "fromIndex"=>0,
                "toIndex"=>100000
            ]);
            if(!empty($responseInfo->return)){
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des contacts échoué. Where : '. json_encode($where). ' dans la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des contacts échoué. Where : '. json_encode($where). ' dans la BDD '.$idDatabase.' Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de compter les contacts avec une condition where
     * @param string $where
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryCountContacts(string $where, int $idDatabase) :mixed
    {
        try {
            $response = $this->magnewsSoap->countContacts([
                'idDatabase' => $idDatabase,
                'query' => MagnewsHelper::makeQuery($where, "COUNT(*)"),
            ]);

            if(isset($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Count des contacts échoué. Where : '. json_encode($where). ' dans la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Count des contacts échoué. Where : '. json_encode($where). ' dans la BDD '.$idDatabase.' Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer les contacts appartenant à une liste d'audience
     * @param int $idSpecialList
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryContactsBelongsToSpecialList(int $idSpecialList, int $idDatabase): mixed
    {
        try {
            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->queryContactsByList([
                'idDatabase' => $idDatabase,
                'idList' => $idSpecialList,
            ]);

            //va chercher les infos des contacts
            $responseInfo = $this->magnewsSoap->fetchContacts([
                'idRowSet'=>$response->return,
                "fromIndex"=>0,
                "toIndex"=>100000
            ]);
            if(!empty($responseInfo->return)){
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des contacts de la liste '. $idSpecialList.' échoué dans la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des contacts de la liste '. $idSpecialList.' échoué dans la BDD '.$idDatabase.' Erreur message : '.$e->getMessage());
        }
    }

    /**
     *Returns the definition for a field in a DB contacts.
     * @param string $id
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDFieldById(string $id, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findFieldByName([
                'idDatabase' =>$idDatabase,
                'name'=> $id
            ]);
            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations du champ '.$id.' de la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations du champ '.$id.' de la BDD '.$idDatabase.' Erreur message : '.$e->getMessage());
        }
    }
    /**
     * Returns basic information about all Databases
     * @return mixed
     * @throws Exception
     */
    public function getBDDsBasicInfos(): mixed
    {
        try {
            $response = $this->magnewsSoap->getAllDatabases([]);
            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations des BDDs',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations des BDDs. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Returns basic information about a Database
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDBasicInfosById(int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findDatabaseById([
                "idDatabase" => $idDatabase
            ]);
            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations de la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations de la BDD '.$idDatabase .'. Erreur message : '.$e->getMessage());
        }
    }
    /**
     * Returns the definition of all fields for a Database
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDFields(int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->getFieldsByDatabase([
                "idDatabase" => $idDatabase
            ]);

            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des champs de la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des champs de la BDD '.$idDatabase. '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Execute a workflow with only id contact variable
     * @param int $idContact
     * @param string $key
     * @param array $option
     * @return bool
     * @throws Exception
     */
    public function enterWorkflowWithIdContact(int $idContact, string $key, array $option = []): bool
    {
        try {
            $response = $this->magnewsSoap->enterWorkflow([
                'contact' =>[
                    "idContact" => $idContact
                ],
                'key' => $key,
                'options' => $option

            ]);
            if(!empty($response->return) && $response->return->idContact >0){
                return true;
            }
            return false;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Execution du workflow échouée. key:'.$key. ' idContact : '.$idContact. ' options: '.json_encode($option),
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Execution du workflow échouée. key:'.$key. ' idContact : '.$idContact. ' options: '.json_encode($option). '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Execute a workflow
     * @param array $contact
     * @param string $key
     * @param array $option
     * @return bool
     * @throws Exception
     */
    public function enterWorkflow(array $contact, string $key, array $option = []): bool
    {
        try {
            $response = $this->magnewsSoap->enterWorkflow([
                'contact' =>$contact,
                'key' => $key,
                'options' => $option

            ]);
            if(!empty($response->return) && $response->return->idContact >0){
                return true;
            }
            return false;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Execution du workflow échouée. key:'.$key. ' Contact : '.json_encode($contact). ' options: '.json_encode($option),
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Execution du workflow échouée. key:'.$key. ' Contact : '.json_encode($contact). ' options: '.json_encode($option). '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Return  "aucun résultat" message
     * @return mixed
     */
    private function noResult(): \Illuminate\Http\JsonResponse
    {
        return response()->json([
            "message" => "Aucun résultat.",
            "status"=> 404
        ]);
    }

    /**
     * return nome utente not exists message
     * @return mixed
     */
    private function nomeUtenteNotExists(): \Illuminate\Http\JsonResponse
    {
        return response()->json([
            "message" => "Ce Nome Utente n'existe pas dans cette base.",
            "status"=> 500
        ]);
    }

    /**
     * Récupère les résultats d'un formulaire
     * @param String $idForm
     * @param DateTime|null $from
     * @param DateTime|null $to
     * @param bool $showAnonymous
     * @return mixed
     * @throws Exception
     */
    public function getFormSubmissionsResults(String $idForm, DateTime $from= null, DateTime $to = null, Bool $showAnonymous = true): mixed
    {
        //TODO : A tester avec un formulaire
        try {
            $params = [
                'idPage' => $idForm,
                'showanonymous'=> $showAnonymous
            ];

            //si une date est renseigné : on ajoute la date aux paramètres
            if(!empty($from)){

                //par defaut : la date d'auourd'hui (si $to n'est pas renseigné)
                if(empty($to)){
                    $to =date('Y-m-d');
                }

                $params = array_merge([
                    'from' => $from,
                    'to' => $to,
                ], $params);
            }

            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->queryFormSubmissions($params);


            //va chercher les infos des résultats
            $responseInfo = $this->magnewsSoap->fetchFormSubmissions([
                'idRowSet'=>$response->return,
                "fromIndex"=>0,
                "toIndex"=>100000
            ]);
            if(!empty($responseInfo->return)){
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des résultats du fromulaire '.$idForm.' échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des résultats du fromulaire '.$idForm.' échoué. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Récupère les résultats d'un survey
     * @param String $idSurvey
     * @param DateTime|null $from
     * @param DateTime|null $to
     * @param bool $showAnonymous
     * @param bool $showUncompleted
     * @return mixed
     * @throws Exception
     */
    public function getSurveyResults(String $idSurvey, DateTime $from= null, DateTime $to = null, Bool $showAnonymous = true, Bool $showUncompleted = true): mixed
    {
        try {
            $params = [
                'idSurvey' => $idSurvey,
                'showanonymous'=> $showAnonymous,
                'showuncompleted'=> $showUncompleted,
            ];

            //si une date est renseigné : on ajoute la date aux paramètres
            if(!empty($from)){

                //par defaut : la date d'auourd'hui (si $to n'est pas renseigné)
                if(empty($to)){
                    $to =date('Y-m-d');
                }

                $params = array_merge([
                    'from' => $from->format('Y-m-d'),
                    'to' => $to->format('Y-m-d'),
                ], $params);
            }

            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->querySurveySessions($params);


            //va chercher les infos des résultats
            $responseInfo = $this->magnewsSoap->fetchSurveySessions([
                'idRowSet'=>$response->return,
                "fromIndex"=>0,
                "toIndex"=>100000
            ]);
            if(!empty($responseInfo->return)){
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des résultats du survey '.$idSurvey.' échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des résultats du survey '.$idSurvey.' échoué. Erreur message : '.$e->getMessage());
        }
    }


    /**
     * Récupère les champs d'un survey
     * @param String $idSurvey
     * @return mixed
     * @throws Exception
     */
    public function getSurveyFields(String $idSurvey): mixed
    {
        try {
            $response = $this->magnewsSoap->getAllFieldsInSurvey([
                'idSurvey' => $idSurvey
            ]);

            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des champs du survey '.$idSurvey.' échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des champs du survey '.$idSurvey.' échoué. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * formate les résultats d'un survey
     * @param string $idSurvey
     * @param DateTime|null $from
     * @param DateTime|null $to
     * @param bool $showAnonymous
     * @param bool $showUncompleted
     * @param bool $needContactFields
     * @param array|null $contactFieldsSelected
     * @param bool $notIgnoreLeadsTest
     * @param bool $notIgnoreSessionsTest
     * @return mixed
     * @throws Exception
     */
    public function getSurveyResultsFormatted(string $idSurvey,
                                              DateTime $from = null,
                                              DateTime $to = null,
                                              Bool $showAnonymous = true,
                                              Bool $showUncompleted = true,
                                              Bool $needContactFields = false,
                                              array $contactFieldsSelected = null,
                                              Bool $notIgnoreLeadsTest = false,
                                              Bool $notIgnoreSessionsTest = false): mixed
    {
        //récupère les résultats du survey
        $results = $this->getSurveyResults($idSurvey, $from, $to, $showAnonymous, $showUncompleted);

        if(empty($results)){
            //pas de résultats
            return [];
        }

        $allData = [];

        //s'il n'y a qu'un seul résultat
        if(!is_object(collect($results)->first())){
            $results = [$results];
        }
        //récurère les questions du survey
        $fields =  $this->getSurveyFields($idSurvey);

        //pour chaque résultat du survey
        foreach (collect($results)->all() as $result) {


            if (empty($notIgnoreSessionsTest) && $this->isTestSessions($result)) {
                //ignore les sessions de tests
                continue;
            }

            //données de session su survey
            $data = [
                "creationDate" => date_create($result->creationDate)->format('Y-m-d H:i:s'),
                "idSession" => $result->idSession,
                "completed" => $result->completed,
                "idContact" =>  $result->idContact ?? null,
                "idSurvey" => $result->idSurvey,
                "lastDate" => date_create($result->lastDate)->format('Y-m-d H:i:s'),
                "submittedPages" => $result->completed ? $result->submittedPages : null,
                "testSession" => $result->testSession,
            ];

            //formatte les réponses aux questions
            if(!empty($result->values)){
                //formate les réponses aux questions
                foreach ($result->values as $response){
                    //ignore les champs sans valeurs ou sans clé
                    if(!isset($response->value) || !isset($response->key)){
                        continue;
                    }
                    $data[$response->key] = $response->value;
                }
            }

            //ajout d'une valeur vide pour les questions non répondues
            foreach ($fields as $field) {
                if (!empty($field->placeholder) && empty($data[$field->placeholder])) {
                    $data[$field->placeholder] = "";
                }
            }

            //si on a besoin des champs contacts, on va les chercher
            if(!empty($needContactFields)){
                $contact = $this->getContactFromIdContact($result->idContact);

                if(empty($notIgnoreLeadsTest)) {
                    // ignore les résultats concernant les leads de tests
                    if ($this->isTestLead($contact->values)) {
                        continue;
                    }
                }

                //formate les champs du contact
                foreach ($contact->values as $entry){
                    //ignore les champs sans valeurs ou sans clé
                    if(!isset($entry->value) || !isset($entry->field)){
                        continue;
                    }
                    //si on a sélectionné des champs
                    if(!empty($contactFieldsSelected)){
                        if(in_array($entry->field, $contactFieldsSelected)){
                            $data[$entry->field] = $entry->value;
                        }
                    }
                    else{
                        //renvoie tous les champs du contacts
                        $data[$entry->field] = $entry->value;
                    }
                }

                //vérifie que toutes les valeurs sélectionnées on été renseignées (sinon on y met une valeur vide) (car magnews ne renvoi pas un champ vide dans la liste des champs du contact)
                foreach ($contactFieldsSelected as $field){
                    if(empty($data[$field])){
                        $data[$field] = "";
                    }
                }

            }
            $allData[] = $data; //ajoute le résultat au tableau retourné
        }

        return $allData;
    }

    /**
     * Fonction permettant d'abonner un contact à une liste spéciale dans Magnews
     * @param integer $idDatabase
     * @param int $idList
     * @param $primaryKeyValue
     * @param null $idSubscriptionCause
     * @return mixed
     * @throws Exception
     */
    public function subscribeContactToList(int $idDatabase, int $idList, $primaryKeyValue, $idSubscriptionCause = null ): mixed
    {
        try {
            $response = $this->magnewsSoap->subscribeContactToList([
                'idDatabase' => $idDatabase,
                'idList'          => $idList,
                'primaryKeyValue' => $primaryKeyValue,
                'idSubscriptionCause'     =>$idSubscriptionCause,
            ]);

            //success
            Log::info(
                __METHOD__ . ' : Abonnement du contact réussi ' . json_encode($response->return). ' dans la liste ' . $idList
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Abonnement du contact échoué. primaryKeyValue: ' . $primaryKeyValue. ' dans la BDD ' . $idDatabase. ' et de la liste '. $idList,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Abonnement du contact échoué. primaryKeyValue: ' . $primaryKeyValue. ' dans la BDD ' . $idDatabase. ' et de la liste '. $idList. '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * @param string $query
     * @return mixed
     * @throws Exception
     */
    public function executeQuery(string $query): mixed
    {
        try {
            $response = $this->magnewsSoap->executeSimpleQuery([
                "query" => $query
            ]);

            if (empty($response->return)) {
                return $this->noResult();
            }

            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return)
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * test si un lead est un test
     * @param $values
     * @return bool
     */
    private function isTestLead($values): bool
    {
        if($isTest =collect($values)->firstWhere('field', 'IS_TEST')){
            return $isTest->value;
        }
        return false;
    }

    /**
     * test si la session est une session de test
     * @param mixed $result
     * @return bool
     */
    private function isTestSessions(mixed $result) : bool
    {
        return $result->testSession;
    }
}
