<?php

namespace Bloom\LaravelMagnews\Services;

use Bloom\LaravelMagnews\DOT\Credentials;
use Bloom\LaravelMagnews\Helpers\MagnewsHelper;
use DateTime;
use Exception;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\JsonResponse;

class MagnewsService
{
    protected MagnewsSoap $magnewsSoap;
    const WEEK_DAYS = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];

    /**
     * @throws \SoapFault
     */
    public function __construct()
    {
        $this->magnewsSoap = new MagnewsSoap(config('magnews.url'));
        $credentials = new Credentials();
        $credentials->password = config('magnews.key');
        $this->magnewsSoap->setCredentials($credentials);
    }

    /**
     * Fonction permettant d'enregistrer un contact dans Magnews
     * @param array $contact
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function saveContact(array $contact, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->mergeContact([
                'idDatabase' => $idDatabase,
                'values' => MagnewsHelper::formatContactValues($contact),
            ]);

            Log::info(
                __METHOD__ . ' : Envoi vers Magnews du contact réussi ' . json_encode($response->return) . ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Envoi vers Magnews du contact échoué ' . json_encode($contact) . ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Envoi vers Magnews du contact échoué ' . json_encode($contact) . ' dans la BDD ' . $idDatabase . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Fonction permettant de désabonner un contact dans Magnews
     * @param string $nomeUtente
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function unsubscribeContact(string $nomeUtente, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->unsubscribeContact([
                'idDatabase' => $idDatabase,
                'values' => MagnewsHelper::createContactValues('NOME_UTENTE', $nomeUtente),
            ]);

            if ($response->return->actionResult == "insert") {
                //le contact n'existe pas. La fonction  subscribeContact essaie de l'insérer
                return $this->nomeUtenteNotExists();
            }

            //success
            Log::info(
                __METHOD__ . ' : Désabonnement du contact réussi ' . json_encode($response->return) . ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Désabonnement du contact échoué. Nome Utente: ' . $nomeUtente . ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Désabonnement du contact échoué. Nome Utente: ' . $nomeUtente . ' dans la BDD ' . $idDatabase . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Fonction permettant d'abonner un contact dans Magnews
     * @param string $nomeUtente
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function subscribeContact(string $nomeUtente, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->subscribeContact([
                'idDatabase' => $idDatabase,
                'values' => MagnewsHelper::createContactValues('NOME_UTENTE', $nomeUtente),
            ]);

            if ($response->return->actionResult == "insert") {
                //le contact n'existe pas. La fonction  subscribeContact essaie de l'insérer
                return $this->nomeUtenteNotExists();
            }

            //success
            Log::info(
                __METHOD__ . ' : Abonnement du contact réussi ' . json_encode($response->return) . ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Abonnement du contact échoué. Nome Utente: ' . $nomeUtente . ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Abonnement du contact échoué. Nome Utente: ' . $nomeUtente . ' dans la BDD ' . $idDatabase . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer un contact dans Magnews via clé primaire
     * @param string $primaryKey
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getContactFromPrimaryKey(string $primaryKey, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findContactByPrimaryKey([
                'idDatabase' => $idDatabase,
                'primaryKeyValue' => $primaryKey,
            ]);

            if (!empty($response->return)) {
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération du contact échoué. Primary key:' . $primaryKey . ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération du contact échoué. Primary key: ' . $primaryKey . ' dans la BDD ' . $idDatabase . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer un contact dans Magnews via id contact
     * @param string $idContact
     * @return mixed
     * @throws Exception
     */
    public function getContactFromIdContact(string $idContact): mixed
    {
        try {
            $response = $this->magnewsSoap->findContactById([
                'idContact' => $idContact,
            ]);

            if (!empty($response->return)) {
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération du contact échoué. idContact: ' . $idContact,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération du contact échoué. idContact: ' . $idContact . ' Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Fonction permettant de faire une requête  sur les contacts en récupérant tous les champs
     * @param string $where
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryContactsAllFields(string $where, int $idDatabase): mixed
    {
        try {
            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->queryContacts([
                'idDatabase' => $idDatabase,
                'query' => MagnewsHelper::makeQuery($where),
            ]);

            //va chercher les infos des contacts
            $responseInfo = $this->magnewsSoap->fetchContacts([
                'idRowSet' => $response->return,
                "fromIndex" => 0,
                "toIndex" => 100000
            ]);
            if (!empty($responseInfo->return)) {
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des contacts échoué. Where : ' . json_encode($where) . ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des contacts échoué. Where : ' . json_encode($where) . ' dans la BDD ' . $idDatabase . ' Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Fonction permettant de compter les contacts avec une condition where
     * @param string $where
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryCountContacts(string $where, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->countContacts([
                'idDatabase' => $idDatabase,
                'query' => MagnewsHelper::makeQuery($where, "COUNT(*)"),
            ]);

            if (isset($response->return)) {
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Count des contacts échoué. Where : ' . json_encode($where) . ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Count des contacts échoué. Where : ' . json_encode($where) . ' dans la BDD ' . $idDatabase . ' Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer les contacts appartenant à une liste d'audience
     * @param int $idSpecialList
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryContactsBelongsToSpecialList(int $idSpecialList, int $idDatabase): mixed
    {
        try {
            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->queryContactsByList([
                'idDatabase' => $idDatabase,
                'idList' => $idSpecialList,
            ]);

            //va chercher les infos des contacts
            $responseInfo = $this->magnewsSoap->fetchContacts([
                'idRowSet' => $response->return,
                "fromIndex" => 0,
                "toIndex" => 100000
            ]);
            if (!empty($responseInfo->return)) {
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des contacts de la liste ' . $idSpecialList . ' échoué dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des contacts de la liste ' . $idSpecialList . ' échoué dans la BDD ' . $idDatabase . ' Erreur message : ' . $e->getMessage());
        }
    }

    /**
     *Returns the definition for a field in a DB contacts.
     * @param string $id
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDFieldById(string $id, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findFieldByName([
                'idDatabase' => $idDatabase,
                'name' => $id
            ]);
            if (!empty($response->return)) {
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations du champ ' . $id . ' de la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations du champ ' . $id . ' de la BDD ' . $idDatabase . ' Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Returns basic information about all Databases
     * @return mixed
     * @throws Exception
     */
    public function getBDDsBasicInfos(): mixed
    {
        try {
            $response = $this->magnewsSoap->getAllDatabases([]);
            if (!empty($response->return)) {
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations des BDDs',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations des BDDs. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Returns basic information about a Database
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDBasicInfosById(int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findDatabaseById([
                "idDatabase" => $idDatabase
            ]);
            if (!empty($response->return)) {
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations de la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations de la BDD ' . $idDatabase . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Returns the definition of all fields for a Database
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDFields(int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->getFieldsByDatabase([
                "idDatabase" => $idDatabase
            ]);

            if (!empty($response->return)) {
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des champs de la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des champs de la BDD ' . $idDatabase . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Execute a workflow with only id contact variable
     * @param int $idContact
     * @param string $key
     * @param array $option
     * @return bool
     * @throws Exception
     */
    public function enterWorkflowWithIdContact(int $idContact, string $key, array $option = []): bool
    {
        try {
            $response = $this->magnewsSoap->enterWorkflow([
                'contact' => [
                    "idContact" => $idContact
                ],
                'key' => $key,
                'options' => $option

            ]);
            if (!empty($response->return) && $response->return->idContact > 0) {
                return true;
            }
            return false;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Execution du workflow échouée. key:' . $key . ' idContact : ' . $idContact . ' options: ' . json_encode($option),
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Execution du workflow échouée. key:' . $key . ' idContact : ' . $idContact . ' options: ' . json_encode($option) . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Execute a workflow
     * @param array $contact
     * @param string $key
     * @param array $option
     * @return bool
     * @throws Exception
     */
    public function enterWorkflow(array $contact, string $key, array $option = []): bool
    {
        try {
            $response = $this->magnewsSoap->enterWorkflow([
                'contact' => $contact,
                'key' => $key,
                'options' => $option

            ]);
            if (!empty($response->return) && $response->return->idContact > 0) {
                return true;
            }
            return false;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Execution du workflow échouée. key:' . $key . ' Contact : ' . json_encode($contact) . ' options: ' . json_encode($option),
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Execution du workflow échouée. key:' . $key . ' Contact : ' . json_encode($contact) . ' options: ' . json_encode($option) . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Return  "aucun résultat" message
     * @return mixed
     */
    private function noResult(): JsonResponse
    {
        return response()->json([
            "message" => "Aucun résultat.",
            "status" => 404
        ]);
    }

    /**
     * return nome utente not exists message
     * @return mixed
     */
    private function nomeUtenteNotExists(): JsonResponse
    {
        return response()->json([
            "message" => "Ce Nome Utente n'existe pas dans cette base.",
            "status" => 500
        ]);
    }

    /**
     * Récupère les résultats d'un formulaire
     * @param String $idForm
     * @param DateTime|null $from
     * @param DateTime|null $to
     * @param bool $showAnonymous
     * @return mixed
     * @throws Exception
     */
    public function getFormSubmissionsResults(string $idForm, DateTime $from = null, DateTime $to = null, bool $showAnonymous = true): mixed
    {
        //TODO : A tester avec un formulaire
        try {
            $params = [
                'idPage' => $idForm,
                'showanonymous' => $showAnonymous
            ];

            //si une date est renseigné : on ajoute la date aux paramètres
            if (!empty($from)) {

                //par defaut : la date d'auourd'hui (si $to n'est pas renseigné)
                if (empty($to)) {
                    $to = date('Y-m-d');
                }

                $params = array_merge([
                    'from' => $from,
                    'to' => $to,
                ], $params);
            }

            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->queryFormSubmissions($params);


            //va chercher les infos des résultats
            $responseInfo = $this->magnewsSoap->fetchFormSubmissions([
                'idRowSet' => $response->return,
                "fromIndex" => 0,
                "toIndex" => 100000
            ]);
            if (!empty($responseInfo->return)) {
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des résultats du fromulaire ' . $idForm . ' échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des résultats du fromulaire ' . $idForm . ' échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Récupère les résultats d'un survey
     * @param String $idSurvey
     * @param DateTime|null $from
     * @param DateTime|null $to
     * @param bool $showAnonymous
     * @param bool $showUncompleted
     * @return mixed
     * @throws Exception
     */
    public function getSurveyResults(string $idSurvey, DateTime $from = null, DateTime $to = null, bool $showAnonymous = true, bool $showUncompleted = true): mixed
    {
        try {
            $params = [
                'idSurvey' => $idSurvey,
                'showanonymous' => $showAnonymous,
                'showuncompleted' => $showUncompleted,
            ];

            //si une date est renseigné : on ajoute la date aux paramètres
            if (!empty($from)) {

                //par defaut : la date d'auourd'hui (si $to n'est pas renseigné)
                if (empty($to)) {
                    $to = date('Y-m-d');
                }

                $params = array_merge([
                    'from' => $from->format('Y-m-d'),
                    'to' => $to->format('Y-m-d'),
                ], $params);
            }

            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->querySurveySessions($params);


            //va chercher les infos des résultats
            $responseInfo = $this->magnewsSoap->fetchSurveySessions([
                'idRowSet' => $response->return,
                "fromIndex" => 0,
                "toIndex" => 100000
            ]);
            if (!empty($responseInfo->return)) {
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des résultats du survey ' . $idSurvey . ' échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des résultats du survey ' . $idSurvey . ' échoué. Erreur message : ' . $e->getMessage());
        }
    }


    /**
     * Récupère les champs d'un survey
     * @param String $idSurvey
     * @return mixed
     * @throws Exception
     */
    public function getSurveyFields(string $idSurvey): mixed
    {
        try {
            $response = $this->magnewsSoap->getAllFieldsInSurvey([
                'idSurvey' => $idSurvey
            ]);

            if (!empty($response->return)) {
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des champs du survey ' . $idSurvey . ' échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des champs du survey ' . $idSurvey . ' échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * formate les résultats d'un survey
     * @param string $idSurvey
     * @param DateTime|null $from
     * @param DateTime|null $to
     * @param bool $showAnonymous
     * @param bool $showUncompleted
     * @param bool $needContactFields
     * @param array|null $contactFieldsSelected
     * @param bool $notIgnoreLeadsTest
     * @param bool $notIgnoreSessionsTest
     * @param bool $filterResultsByThisWeek
     * @param string $firstDayOfTheWeek : monday,tuesday, wednesday, thursday, friday, saturday, sunday
     * @param bool $filterResultsByLastWeek
     * @param bool $needIdNewsletter
     * @return array
     * @throws Exception
     */
    public function getSurveyResultsFormatted(string   $idSurvey,
                                              DateTime $from = null,
                                              DateTime $to = null,
                                              bool     $showAnonymous = true,
                                              bool     $showUncompleted = true,
                                              bool     $needContactFields = false,
                                              array    $contactFieldsSelected = null,
                                              bool     $notIgnoreLeadsTest = false,
                                              bool     $notIgnoreSessionsTest = false,
                                              bool     $filterResultsByThisWeek = false,
                                              string   $firstDayOfTheWeek = 'monday',
                                              bool     $filterResultsByLastWeek = false,
                                              bool     $needIdNewsletter = true,
    ): array
    {
        if ($filterResultsByThisWeek) {
            $from = date_create($this->getFirstDayOfTheWeek($firstDayOfTheWeek));
            $to = date_create($this->getLastDayOfTheWeek($firstDayOfTheWeek));
        }
        if ($filterResultsByLastWeek) {
            $from = date_create($this->getFirstDayOfLastWeek());
            $to = date_create(date('Y-m-d'));
        }
        //récupère les résultats du survey
        $results = $this->getSurveyResults($idSurvey, $from, $to, $showAnonymous, $showUncompleted);

        if (empty($results) || ($results instanceof JsonResponse && json_decode($results->getContent())->status == 404)) {
            //pas de résultats
            return [];
        }

        $allData = [];

        //s'il n'y a qu'un seul résultat
        if (!is_object(collect($results)->first())) {
            $results = [$results];
        }
        //récurère les questions du survey
        $fields = $this->getSurveyFields($idSurvey);

        //pour chaque résultat du survey
        foreach (collect($results)->all() as $result) {


            if (empty($notIgnoreSessionsTest) && $this->isTestSessions($result)) {
                //ignore les sessions de tests
                continue;
            }
            $creationDate = date_create($result->creationDate);

            //données de session su survey
            $data = [
                "creationDate" => $creationDate->format('Y-m-d H:i:s'),
                "idSession" => $result->idSession,
                "completed" => $result->completed,
                "idContact" => $result->idContact ?? null,
                "idSurvey" => $result->idSurvey,
                "lastDate" => date_create($result->lastDate)->format('Y-m-d H:i:s'),
                "submittedPages" => $result->completed ? $result->submittedPages : null,
                "testSession" => $result->testSession,
            ];

            //formatte les réponses aux questions
            if (!empty($result->values)) {
                foreach ($result->values as $response) {
                    //ignore les champs sans valeurs ou sans clé
                    if (!isset($response->value) || !isset($response->key)) {
                        continue;
                    }
                    $data[$response->key] = $response->value;
                }
            }

            //ajout d'une valeur vide pour les questions non répondues
            foreach ($fields as $field) {
                if (!empty($field->placeholder) && empty($data[$field->placeholder])) {
                    $data[$field->placeholder] = "";
                }
            }

            //si on a besoin des champs contacts, on va les chercher
            if (!empty($needContactFields)) {
                $contact = $this->getContactFromIdContact($result->idContact);

                if (empty($notIgnoreLeadsTest)) {
                    // ignore les résultats concernant les leads de tests
                    if ($this->isTestLead($contact->values)) {
                        continue;
                    }
                }

                //formate les champs du contact
                foreach ($contact->values as $entry) {
                    //ignore les champs sans valeurs ou sans clé
                    if (!isset($entry->value) || !isset($entry->field)) {
                        continue;
                    }
                    //si on a sélectionné des champs
                    if (!empty($contactFieldsSelected)) {
                        if (in_array($entry->field, $contactFieldsSelected)) {
                            $data[$entry->field] = $entry->value;
                        }
                    } else {
                        //renvoie tous les champs du contacts
                        $data[$entry->field] = $entry->value;
                    }
                }

                //vérifie que toutes les valeurs sélectionnées on été renseignées (sinon on y met une valeur vide) (car magnews ne renvoi pas un champ vide dans la liste des champs du contact)
                foreach ($contactFieldsSelected as $field) {
                    if (empty($data[$field])) {
                        $data[$field] = "";
                    }
                }

            }

            //si besoin : on va chercher l'id newsletter correspondant au mail du sondage
            if (!empty($result->idContact) && $needIdNewsletter) {
                $idsNewsletter = [];

                //récupère les messages reçus du contact et filtre par ceux qui contiennent le lien du survey
                $messages = $this->getSimpleMessageEvents(null, $result->idContact);
                $idMessages = collect($messages)->filter(function ($message) use ($idSurvey) {
                    //filtre que les messages avec le survey
                    return !empty($message->link) && str_contains($message->link, 'survey id=' . $idSurvey);
                })
                    ->map(function ($message) {
                        //récupère que l'idSimpleMessage
                        return $message->idSimpleMessage;
                    })->flatten()->all();

                $idMessagesJoined = join(',', $idMessages);

                //get id newsletter for each message
                $events = $this->executeQuery("select idnewsletter from SIMPLEMESSAGES where idsimplemessage  in (" . $idMessagesJoined . ")");

                //format ids newsletter
                if (is_array($events->record)) {
                    //many rows
                    foreach ($events->record as $record) {
                        $idsNewsletter[] = $record->cell->value;
                    }
                } else {
                    // only one row
                    $idsNewsletter[] = $events->record->cell->value;
                }
                $data['newsletters'] = join(',', $idsNewsletter);
            }
            $allData[] = $data; //ajoute le résultat au tableau retourné
        }

        return $allData;
    }

    /**
     * Fonction permettant d'abonner un contact à une liste spéciale dans Magnews
     * @param integer $idDatabase
     * @param int $idList
     * @param $primaryKeyValue
     * @param null $idSubscriptionCause
     * @return mixed
     * @throws Exception
     */
    public function subscribeContactToList(int $idDatabase, int $idList, $primaryKeyValue, $idSubscriptionCause = null): mixed
    {
        try {
            $response = $this->magnewsSoap->subscribeContactToList([
                'idDatabase' => $idDatabase,
                'idList' => $idList,
                'primaryKeyValue' => $primaryKeyValue,
                'idSubscriptionCause' => $idSubscriptionCause,
            ]);

            //success
            Log::info(
                __METHOD__ . ' : Abonnement du contact réussi ' . json_encode($response->return) . ' dans la liste ' . $idList
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Abonnement du contact échoué. primaryKeyValue: ' . $primaryKeyValue . ' dans la BDD ' . $idDatabase . ' et de la liste ' . $idList,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Abonnement du contact échoué. primaryKeyValue: ' . $primaryKeyValue . ' dans la BDD ' . $idDatabase . ' et de la liste ' . $idList . '. Erreur message : ' . $e->getMessage());
        }
    }


    /**
     * Fonction qui récupère les infos de base d'une compagne
     * @param int $idCampaign
     * @return mixed
     * @throws Exception
     */
    public function findCampaignById(int $idCampaign): mixed
    {
        try {
            $response = $this->magnewsSoap->findCampaignById([
                'idCampaign' => $idCampaign
            ]);

            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return) . ' dans la compagne ' . $idCampaign
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué. id compaign: ' . $idCampaign,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. id compaign: ' . $idCampaign . '. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * Fonction qui récupère les infos de base d'une newsletter
     * @param int $idNewsletter
     * @return mixed
     * @throws Exception
     */
    public function findNewsletterById(int $idNewsletter): mixed
    {
        try {
            $response = $this->magnewsSoap->findNewsletterById([
                'idNewsletter' => $idNewsletter
            ]);

            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return) . ' dans la newsletter ' . $idNewsletter
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué. id newsletter: ' . $idNewsletter,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. id newsletter: ' . $idNewsletter . '. Erreur message : ' . $e->getMessage());
        }
    }


    /**
     * @param int|null $idSimpleMessage
     * @param int|null $idContact
     * @param int|null $idMessageType
     * @param int|null $idMessageCategory
     * @param string|null $from
     * @param string|null $to
     * @param string|null $channel
     * @return mixed
     * @throws Exception
     */
    public function getSimpleMessageEvents(
        int    $idSimpleMessage = null,
        int    $idContact = null,
        int    $idMessageType = null,
        int    $idMessageCategory = null,
        string $from = null,
        string $to = null,
        string $channel = null): mixed
    {
        try {
            $options = [];
            if (!empty($idSimpleMessage)) {
                $options[] = ['key' => 'idsimplemessage', 'value' => $idSimpleMessage];
            }
            if (!empty($idContact)) {
                $options[] = ['key' => 'idcontact', 'value' => $idContact];
            }

            if (!empty($idMessageType)) {
                $options[] = ['key' => 'idmessagetype', 'value' => $idMessageType];
            }

            if (!empty($idMessageCategory)) {
                $options[] = ['key' => 'idmessagecategory', 'value' => $idMessageCategory];
            }
            if (!empty($from)) {
                $options[] = ['key' => 'from', 'value' => $from];
            }

            if (!empty($to)) {
                $options[] = ['key' => 'to', 'value' => $to];
            }
            if (!empty($channel)) {
                $options[] = ['key' => 'channel', 'value' => $channel];
            }
            $response = $this->magnewsSoap->getSimpleMessageEvents([
                'options' => $options
            ]);

            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return)
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * @param int $idNewsletter
     * @return mixed
     * @throws Exception
     */
    public function getDeliveryStatusForNewsletter(int $idNewsletter): mixed
    {
        try {
            $response = $this->magnewsSoap->getDeliveryStatusForNewsletter([
                "idNewsletter" => $idNewsletter
            ]); //vide
            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return)
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * @param int $idCampaign
     * @return mixed
     * @throws Exception
     */
    public function queryDeliveryStatusForCampaign(int $idCampaign): mixed
    {
        try {
            $response = $this->magnewsSoap->queryDeliveryStatus([
                "idCampaign" => $idCampaign
            ]);
            if (empty($response->return)) {
                return $this->noResult();
            }

            $response = $this->magnewsSoap->fetchDeliveryStatus([
                "idRowSet" => $response->return,
                "fromIndex" => 0,
                "toIndex" => 1000000
            ]);

            if (empty($response->return)) {
                return $this->noResult();
            }
            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return)
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * @param int $idNewsletter
     * @return mixed
     * @throws Exception
     */
    public function getBasicNewsletterReport(int $idNewsletter): mixed
    {
        try {
            $response = $this->magnewsSoap->getBasicNewsletterReport([
                "idNewsletter" => $idNewsletter
            ]);
            if (empty($response)) {
                return $this->noResult();
            }

            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return)
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * @param int $idNewsletter
     * @param array $options
     * @return mixed
     * @throws Exception
     */
    public function queryDetailedNewsletterReport(int $idNewsletter, array $options = []): mixed
    {
        try {
            //TODO $options à customiser
            $response = $this->magnewsSoap->queryDetailedNewsletterReport([
                "idNewsletter" => $idNewsletter,
                'options' => [
                    ['key' => 'bouncesdetails', 'value' => true],
                    ['key' => 'clientinfo', 'value' => true],
                    ['key' => 'batchesdetail', 'value' => true],
                    ['key' => 'unsubscribeinfo', 'value' => true],
                    ['key' => 'contentsdetails', 'value' => true],
                    ['key' => 'events', 'value' => true],
                ]
            ]);

            if (empty($response->return)) {
                return $this->noResult();
            }

            $response = $this->magnewsSoap->fetchDetailedNewsletterReport([
                "idRowSet" => $response->return,
                "fromIndex" => 0,
                "toIndex" => 10
            ]);

            if (empty($response->return)) {
                return $this->noResult();
            }

            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return)
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * @param int $idContact
     * @param array $options
     * @return mixed
     * @throws Exception
     */
    public function queryDetailedContactReport(int $idContact, array $options = []): mixed
    {
        try {
            //TODO $options à customiser
            $response = $this->magnewsSoap->queryDetailedContactReport([
                "idContact" => $idContact,
                'options' => [
                    ['key' => 'bouncesdetails', 'value' => true],
                    ['key' => 'clientinfo', 'value' => true],
                    ['key' => 'unsubscribeinfo', 'value' => true],
                    ['key' => 'contentsdetails', 'value' => true],
                    ['key' => 'events', 'value' => true],
                ]
            ]);

            if (empty($response->return)) {
                return $this->noResult();
            }

            $response = $this->magnewsSoap->fetchDetailedContactReport([
                "idRowSet" => $response->return,
                "fromIndex" => 0,
                "toIndex" => 10
            ]);

            if (empty($response->return)) {
                return $this->noResult();
            }
            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return)
            );
            return $response->return;
        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * @param string $query
     * @return mixed
     * @throws Exception
     */
    public function executeQuery(string $query): mixed
    {
        try {
            $response = $this->magnewsSoap->executeSimpleQuery([
                "query" => $query
            ]);

            if (empty($response->return)) {
                return $this->noResult();
            }

            //success
            Log::info(
                __METHOD__ . ' : récupération réussi ' . json_encode($response->return)
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : récupération échoué.',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : récupération échoué. Erreur message : ' . $e->getMessage());
        }
    }

    /**
     * test si un lead est un test
     * @param $values
     * @return bool
     */
    private function isTestLead($values): bool
    {
        if ($isTest = collect($values)->firstWhere('field', 'IS_TEST')) {
            return $isTest->value;
        }
        return false;
    }

    /**
     * test si la session est une session de test
     * @param mixed $result
     * @return bool
     */
    private function isTestSessions(mixed $result): bool
    {
        return $result->testSession ?? false;
    }

    /**
     * fonction qui récupère le premier jour de cette semaine (en fonction du jour en paramètre)
     * @param string $firstDayOfTheWeek
     * @return string
     * @throws Exception
     */
    private function getFirstDayOfTheWeek(string $firstDayOfTheWeek): string
    {
        if (in_array(strtolower($firstDayOfTheWeek), self::WEEK_DAYS)) {
            return date('Y-m-d', strtotime("$firstDayOfTheWeek this week"));
        }
        throw new Exception("Le jour de la semaine n'est pas correcte. Valeurs attendues: " . implode(self::WEEK_DAYS));
    }

    /**
     * function qui récupère la date de J-7
     * @return string
     */
    private function getFirstDayOfLastWeek(): string
    {
        return date('Y-m-d', strtotime(date('l') . " last week"));
    }

    /**
     * fonction qui récupère le dernier jour de cette semaine (en fonction du jour en paramètre)
     * @param string $firstDayOfTheWeek
     * @return string
     * @throws Exception
     */
    private function getLastDayOfTheWeek(string $firstDayOfTheWeek): string
    {
        if (!in_array(strtolower($firstDayOfTheWeek), self::WEEK_DAYS)) {
            throw new Exception("Le jour de la semaine n'est pas correcte. Valeurs attendues: " . implode(self::WEEK_DAYS));
        }

        $lastDay = match ($firstDayOfTheWeek) {
            'tuesday' => 'monday',
            'wednesday' => 'tuesday',
            'thursday' => 'wednesday',
            'friday' => 'thursday',
            'saturday' => 'friday',
            'sunday' => 'saturday',
            default => 'sunday',
        };

        return date('Y-m-d', strtotime("$lastDay this week"));
    }
}
