<?php

namespace Bloom\LaravelMagnews\Services;

use Bloom\LaravelMagnews\DOT\Credentials;
use Bloom\LaravelMagnews\Helpers\MagnewsHelper;
use Exception;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class MagnewsService
{
    protected MagnewsSoap $magnewsSoap;
    protected Credentials $credentials;

    /**
     * @throws \SoapFault
     */
    public function __construct()
    {
        $this->magnewsSoap = new MagnewsSoap(config('magnews.url'));
        $credentials = new Credentials();
        $credentials->password = config('magnews.key');
        $this->magnewsSoap->setCredentials($credentials);
    }

    /**
     * Fonction permettant d'enregistrer un contact dans Magnews
     * @param array $contact
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function saveContact(Array $contact, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->mergeContact([
                'idDatabase' => $idDatabase,
                'values'     => MagnewsHelper::formatContactValues($contact),
            ]);

            Log::info(
                __METHOD__ . ' : Envoi vers Magnews du contact réussi ' . json_encode($response->return). ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Envoi vers Magnews du contact échoué ' . json_encode($contact). ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Envoi vers Magnews du contact échoué ' . json_encode($contact). ' dans la BDD ' . $idDatabase . '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de désabonner un contact dans Magnews
     * @param string $nomeUtente
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function unsubscribeContact(string $nomeUtente, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->unsubscribeContact([
                'idDatabase' => $idDatabase,
                'values'     => MagnewsHelper::createContactValues('NOME_UTENTE',$nomeUtente),
            ]);

            if($response->return->actionResult == "insert"){
                //le contact n'existe pas. La fonction  subscribeContact essaie de l'insérer
                return $this->nomeUtenteNotExists();
            }

            //success
            Log::info(
                __METHOD__ . ' : Désabonnement du contact réussi ' . json_encode($response->return). ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Désabonnement du contact échoué. Nome Utente: ' . $nomeUtente. ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Désabonnement du contact échoué. Nome Utente: ' . $nomeUtente. ' dans la BDD ' . $idDatabase . '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant d'abonner un contact dans Magnews
     * @param string $nomeUtente
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function subscribeContact(string $nomeUtente, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->subscribeContact([
                'idDatabase' => $idDatabase,
                'values'     => MagnewsHelper::createContactValues('NOME_UTENTE',$nomeUtente),
            ]);

            if($response->return->actionResult == "insert"){
                //le contact n'existe pas. La fonction  subscribeContact essaie de l'insérer
                return $this->nomeUtenteNotExists();
            }

            //success
            Log::info(
                __METHOD__ . ' : Abonnement du contact réussi ' . json_encode($response->return). ' dans la BDD ' . $idDatabase
            );
            return $response->return;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Abonnement du contact échoué. Nome Utente: ' . $nomeUtente. ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Abonnement du contact échoué. Nome Utente: ' . $nomeUtente. ' dans la BDD ' . $idDatabase . '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer un contact dans Magnews via clé primaire
     * @param string $primaryKey
     * @param integer $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getContactFromPrimaryKey(string $primaryKey, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findContactByPrimaryKey([
                'idDatabase' => $idDatabase,
                'primaryKeyValue' => $primaryKey,
            ]);

            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération du contact échoué. Primary key:' . $primaryKey. ' dans la BDD ' . $idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération du contact échoué. Primary key: ' .$primaryKey. ' dans la BDD ' . $idDatabase . '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer un contact dans Magnews via id contact
     * @param string $idContact
     * @return mixed
     * @throws Exception
     */
    public function getContactFromIdContact(string $idContact): mixed
    {
        try {
            $response = $this->magnewsSoap->findContactById([
                'idContact' => $idContact,
            ]);

            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération du contact échoué. idContact: '.$idContact,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération du contact échoué. idContact: '.$idContact.' Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de faire une requête  sur les contacts en récupérant tous les champs
     * @param string $where
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryContactsAllFields(string $where, int $idDatabase): mixed
    {
        try {
            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->queryContacts([
                'idDatabase' => $idDatabase,
                'query' => MagnewsHelper::makeQuery($where),
            ]);

            //va chercher les infos des contacts
            $responseInfo = $this->magnewsSoap->fetchContacts([
                'idRowSet'=>$response->return,
                "fromIndex"=>0,
                "toIndex"=>100000
            ]);
            if(!empty($responseInfo->return)){
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des contacts échoué. Where : '. json_encode($where). ' dans la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des contacts échoué. Where : '. json_encode($where). ' dans la BDD '.$idDatabase.' Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de compter les contacts avec une condition where
     * @param string $where
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryCountContacts(string $where, int $idDatabase) :mixed
    {
        try {
            $response = $this->magnewsSoap->countContacts([
                'idDatabase' => $idDatabase,
                'query' => MagnewsHelper::makeQuery($where, "COUNT(*)"),
            ]);

            if(isset($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Count des contacts échoué. Where : '. json_encode($where). ' dans la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Count des contacts échoué. Where : '. json_encode($where). ' dans la BDD '.$idDatabase.' Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Fonction permettant de récupérer les contacts appartenant à une liste d'audience
     * @param int $idSpecialList
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function queryContactsBelongsToSpecialList(int $idSpecialList, int $idDatabase): mixed
    {
        try {
            //return a rowSetId pour le résultat de la requête
            $response = $this->magnewsSoap->queryContactsByList([
                'idDatabase' => $idDatabase,
                'idList' => $idSpecialList,
            ]);

            //va chercher les infos des contacts
            $responseInfo = $this->magnewsSoap->fetchContacts([
                'idRowSet'=>$response->return,
                "fromIndex"=>0,
                "toIndex"=>100000
            ]);
            if(!empty($responseInfo->return)){
                return $responseInfo->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des contacts de la liste '. $idSpecialList.' échoué dans la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des contacts de la liste '. $idSpecialList.' échoué dans la BDD '.$idDatabase.' Erreur message : '.$e->getMessage());
        }
    }

    /**
     *Returns the definition for a field in a DB contacts.
     * @param string $id
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDFieldById(string $id, int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findFieldByName([
                'idDatabase' =>$idDatabase,
                'name'=> $id
            ]);
            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations du champ '.$id.' de la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations du champ '.$id.' de la BDD '.$idDatabase.' Erreur message : '.$e->getMessage());
        }
    }
    /**
     * Returns basic information about all Databases
     * @return mixed
     * @throws Exception
     */
    public function getBDDsBasicInfos(): mixed
    {
        try {
            $response = $this->magnewsSoap->getAllDatabases([]);
            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations des BDDs',
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations des BDDs. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Returns basic information about a Database
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDBasicInfosById(int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->findDatabaseById([
                "idDatabase" => $idDatabase
            ]);
            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des informations de la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des informations de la BDD '.$idDatabase .'. Erreur message : '.$e->getMessage());
        }
    }
    /**
     * Returns the definition of all fields for a Database
     * @param int $idDatabase
     * @return mixed
     * @throws Exception
     */
    public function getBDDFields(int $idDatabase): mixed
    {
        try {
            $response = $this->magnewsSoap->getFieldsByDatabase([
                "idDatabase" => $idDatabase
            ]);

            if(!empty($response->return)){
                return $response->return;
            }
            return $this->noResult();

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Récupération des champs de la BDD '.$idDatabase,
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Récupération des champs de la BDD '.$idDatabase. '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Execute a workflow with only id contact variable
     * @param int $idContact
     * @param string $key
     * @param array $option
     * @return bool
     * @throws Exception
     */
    public function enterWorkflowWithIdContact(int $idContact, string $key, array $option = []): bool
    {
        try {
            $response = $this->magnewsSoap->enterWorkflow([
                'contact' =>[
                    "idContact" => $idContact
                ],
                'key' => $key,
                'options' => $option

            ]);
            if(!empty($response->return) && $response->return->idContact >0){
                return true;
            }
            return false;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Execution du workflow échouée. key:'.$key. ' idContact : '.$idContact. ' options: '.json_encode($option),
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Execution du workflow échouée. key:'.$key. ' idContact : '.$idContact. ' options: '.json_encode($option). '. Erreur message : '.$e->getMessage());
        }
    }

    /**
     * Execute a workflow
     * @param array $contact
     * @param string $key
     * @param array $option
     * @return bool
     * @throws Exception
     */
    public function enterWorkflow(array $contact, string $key, array $option = []): bool
    {
        try {
            $response = $this->magnewsSoap->enterWorkflow([
                'contact' =>$contact,
                'key' => $key,
                'options' => $option

            ]);
            if(!empty($response->return) && $response->return->idContact >0){
                return true;
            }
            return false;

        } catch (Exception $e) {
            Log::error(
                __METHOD__ . ' : Execution du workflow échouée. key:'.$key. ' Contact : '.json_encode($contact). ' options: '.json_encode($option),
                ['error' => $e->getMessage()]
            );
            throw new Exception(__METHOD__ . ' : Execution du workflow échouée. key:'.$key. ' Contact : '.json_encode($contact). ' options: '.json_encode($option). '. Erreur message : '.$e->getMessage());
        }
    }


    private function noResult(): \Illuminate\Http\JsonResponse
    {
        return response()->json([
            "message" => "Aucun résultat.",
            "status"=> 404
        ]);
    }
    private function nomeUtenteNotExists(): \Illuminate\Http\JsonResponse
    {
        return response()->json([
            "message" => "Ce Nome Utente n'existe pas dans cette base.",
            "status"=> 500
        ]);
    }
}
