<?php
/**
 * Created by IntelliJ IDEA.
 * User: Loïc Rameau
 * Date: 30/03/2017
 * Time: 12:21
 */

namespace Bloom\Honda\Services;

use Bloom\Honda\Error\HttpSSOException;
use Bloom\Honda\Events\SsoCalled;
use Bloom\Honda\Events\SsoFailed;
use Bloom\Honda\Events\SsoResponseReceived;
use Bloom\Honda\Events\SsoSuccess;
use Bloom\Honda\Model\HondaUser;
use GuzzleHttp\Client;
use SimpleXMLElement;

class HondaSSO
{
    protected Client $client;

    protected string $url;

    protected ?HondaUser $last_user;

    public function __construct(string $url)
    {
        $this->client = new Client();
        $this->url = $url;
    }

    /**
     * Check if the given hash is a valid hash generate by the HONDA SSO
     *
     * @param string $hash The hash to test
     * @param int    $id   The services id
     *
     * @return bool
     * @throws HttpSSOException|\GuzzleHttp\Exception\GuzzleException
     */
    public function check(string $hash, int $id): bool
    {
        $this->verifyHash($hash);
        $url = $this->url . '/' . $id . '/' . $hash;
        $response = $this->client->get($url);
        event(new SsoCalled($url));
        if ($response->getStatusCode() !== 200) {
            event(new SsoFailed('Error coming from sso server (' . $response->getReasonPhrase() . ')'));

            return false;
        }
        $xml = simplexml_load_string($response->getBody()->getContents());

        event(new SsoResponseReceived($xml->saveXML()));
        if ((string)$xml->Status !== 'AUTHENTICATED') {
            event(new SsoFailed('Authentication error (`' . ((string)$xml->FailureReason) . '`)'));

            return false;
        }
        $this->last_user = new HondaUser();
        $this->populateUser($xml, $this->last_user);
        event(new SsoSuccess($this->last_user));

        return true;
    }

    /**
     * Get the last authenticate user
     * @return HondaUser|null
     */
    public function user(): ?HondaUser
    {
        return $this->last_user;
    }

    /**
     * Populate the given user, with the info in the xml
     *
     * @param SimpleXMLElement $xml
     * @param HondaUser         $user
     */
    protected function populateUser(SimpleXMLElement $xml, HondaUser $user): void
    {
        $user->email = (string)$xml->Email;
        $user->country_code = (string)$xml->CountryCode;
        $user->dealer = (string)$xml->BusinessUnit;
        $user->fullname = (string)$xml->Fullname;
        $user->username = (string)$xml->Username;
    }

    /**
     * Check basic thing on the hash
     *
     * @param string $hash The hash to test
     *
     * @return bool
     */
    public function verifyHash(string $hash) : bool
    {
        if (strlen($hash) !== 15) {
            throw new \InvalidArgumentException('SSO Hash must be 15 char long');
        }
        if (!ctype_alnum($hash)) {
            throw new \InvalidArgumentException('SSO Hash must be alpha-numeric');
        }

        return true;
    }
}
