<?php
/**
 * Created by IntelliJ IDEA.
 * User: Loïc Rameau
 * Date: 30/03/2017
 * Time: 12:21
 */

namespace Bloom\Honda\Services;

use Bloom\Honda\Events\SsoCalled;
use Bloom\Honda\Events\SsoFailed;
use Bloom\Honda\Events\SsoSuccess;
use Bloom\Honda\Model\HondaUser;
use Bloom\Honda\Error\HttpSSOException;
use GuzzleHttp\Client;

class HondaSSO
{
    /**
     * @var Client
     */
    protected $client;
    /**
     * @var string
     */
    protected $url;

    /**
     * @var HondaUser
     */
    protected $last_user;

    /**
     * HondaSSO constructor.
     * @param string $url
     */
    public function __construct($url)
    {
        if (!is_string($url)) {
            throw new \InvalidArgumentException('Url must be a string');
        }
        $this->client = new Client();
        $this->url = $url;
    }

    /**
     * Check if the given hash is a valid hash generate by the HONDA SSO
     * @param string $hash The hash to test
     * @param int    $id   The services id
     * @return bool
     * @throws HttpSSOException
     */
    public function check($hash, $id)
    {
        $this->verifyHash($hash);
        $url = $this->url . '/' . $id . '/' . $hash;
        $response = $this->client->get($url);
        event(new SsoCalled($url));
        if ($response->getStatusCode() !== 200) {
            event(new SsoFailed('Error coming from sso server ('.$response->getReasonPhrase().')'));
            return false;
        }
        $xml = simplexml_load_string($response->getBody()->getContents());

        if ((string)$xml->Status !== 'AUTHENTICATED') {
            event(new SsoFailed('Authentication error (`' . ((string)$xml->FailureReason) . '`)'));
            return false;
        }
        $this->last_user = new HondaUser();
        $this->populateUser($xml, $this->last_user);
        event(new SsoSuccess($this->last_user));
        return true;
    }

    /**
     * Get the last authenticate user
     * @return HondaUser
     */
    public function user()
    {
        return $this->last_user;
    }

    /**
     * Populate the given user, with the info in the xml
     * @param \SimpleXMLElement $xml
     * @param HondaUser         $user
     */
    protected function populateUser(\SimpleXMLElement $xml, HondaUser $user)
    {
        $user->email = (string)$xml->Email;
        $user->country_code = (string)$xml->CountryCode;
        $user->dealer = (string)$xml->BusinessUnit;
        $user->fullname = (string)$xml->Fullname;
        $user->username = (string)$xml->Username;
    }

    /**
     * Check basic thing on the hash
     * @param string $hash The hash to test
     * @return bool
     */
    protected function verifyHash($hash)
    {
        if (strlen($hash) !== 15) {
            throw new \InvalidArgumentException('SSO Hash must be 15 char long');
        }
        if (!ctype_alnum($hash)) {
            throw new \InvalidArgumentException('SSO Hash must be alpha-numeric');
        }
        return true;
    }
}
