<?php
/**
 * Created by IntelliJ IDEA.
 * User: bloom
 * Date: 18/01/2016
 * Time: 16:02
 */

namespace Bloom\Entis\Api\Model;

use Bloom\Entis\Api\Error\Database\InsertError;
use Bloom\Entis\Api\Error\Database\UpdateError;
use Bloom\Entis\Api\Error\Login\InvalidEntityError;
use Bloom\Entis\Api\Error\Login\InvalidStatusError;
use Bloom\Entis\Api\Error\Login\InvalidUserPasswordError;
use Bloom\Entis\Api\Error\Login\UserBlockedError;
use Bloom\Entis\Api\Middleware\Auth;

class User extends Model
{
    const BLOCK_DELAY = 30;

    protected static $table = 'cms_utilisateurs';

    protected static $exclude_single =
        [
            'descUtilisateur','metierUtilisateur','villeUtilisateur','numconf',
            'dateutil','autolog', 'expire', 'passUtilisateurMD5', 'passUtilisateur',
            'imgUtilisateur'
        ];

    public $id_utilisateur;
    public $nomUtilisateur;
    public $prenomUtilisateur;
    public $imgUtilisateur;
    public $loginUtilisateur;
    public $passUtilisateur;
    public $mailUtilisateur;
    public $siteUtilisateur;
    public $urlUtilisateur;
    public $descUtilisateur;
    public $metierUtilisateur;
    public $villeUtilisateur;
    public $statutVisiteur;
    public $statutGestionnaire;
    public $user_section;
    public $user_entite;
    public $numconf;
    public $dateutil;
    public $autolog;
    public $expire;
    public $last_vist;
    public $passUtilisateurMD5;

    /**
     * @param $login
     * @param $password
     * @return array
     */
    public static function Login($login, $password) {
        /**
         * @var User $user
         */
        $user = static::getItem(
            [
                ['column'=>'loginUtilisateur', 'operator'=> '=', 'value' => $login]
            ]
        );
        if(!$user) {
            return
                [
                    'code'=> 406,
                    'message'=>'Unauthorized. Invalid login & password couples.',
                    'type'=> 'Bloom\Entis\Api\Error\Login\InvalidUserPasswordError',
                    'data'=> [
                        'file' => __FILE__,
                        'line' => __LINE__
                    ]
                ];
        }
        try
        {
            $user->verifyUserBlocked();
            $user->verifyPassword($password);
            $user->verifyEntity();
            $user->verifyCodeStatus();
        }
        catch(InvalidUserPasswordError $e) {
            $user->attemptFailed();
            return
                [
                    'code'=> 406,
                    'message'=>$e->getMessage(),
                    'type'=> get_class($e),
                    'data'=> [
                        'file' => $e->getFile(),
                        'line' => $e->getLine()
                    ]
                ];
        }
        catch(InvalidStatusError $e) {
            $user->attemptFailed();
            return
                [
                    'code'=> 406,
                    'message'=>$e->getMessage(),
                    'type'=> get_class($e),
                    'data'=> [
                        'file' => $e->getFile(),
                        'line' => $e->getLine()
                    ]
                ];
        }
        catch(InvalidEntityError $e) {
            $user->attemptFailed();
            return
                [
                    'code'=> 406,
                    'message'=>$e->getMessage(),
                    'type'=> get_class($e),
                    'data'=> [
                        'file' => $e->getFile(),
                        'line' => $e->getLine()
                    ]
                ];
        }
        catch(UserBlockedError $e) {
            $user->attemptFailed();
            return
                [
                    'code'=> 406,
                    'message'=>$e->getMessage(),
                    'type'=> get_class($e),
                    'data'=> [
                        'file' => $e->getFile(),
                        'line' => $e->getLine()
                    ]
                ];
        }
        catch(\Exception $e) {
            return
                [
                    'code' => 500,
                    'message' => 'Unhandled exception',
                    'data' => [
                        'type'=> get_class($e),
                        'code' => $e->getCode(),
                        'message'=> $e->getMessage(),
                        'file' => $e->getFile(),
                        'line' => $e->getLine(),
                        'trace' => $e->getTrace(),
                        'previous' => $e->getPrevious()
                    ]
                ];
        }

        $user->clearAttempt();

        // Tout est OK
        return [
            'code' => 200,
            'message' => 'Login successful',
            'data' => User::getItem([
                ['column'=>'loginUtilisateur', 'operator'=> '=', 'value' => $login]
            ])->filter(true)
        ];
    }

    protected function verifyPassword($password)
    {
        if($this->passUtilisateurMD5 !== $password) {
            throw new InvalidUserPasswordError('Unauthorized. Invalid login & password couples.');
        }
    }

    protected function verifyEntity()
    {
        if($this->user_entite === null) {
            return;
        }

        if($this->user_entite !== Auth::currentEntity()->code) {
            throw new InvalidEntityError('Unauthorized. Invalid entity couples');
        }
    }

    protected function verifyCodeStatus()
    {
        $statut = static::$db->select()->from('cms_statuts')->where('nivo_statut','=',$this->statutGestionnaire)->execute()->fetchObject();
        if($statut === false || ($statut->exig_statut !== null && $this->{$statut->exig_statut} === null)) {
            throw new InvalidStatusError('Unauthorized. Insufficient access right');
        }
    }

    protected function verifyUserBlocked()
    {
        $db = static::$db;

        $secure = $db->select()->from('cms_secure')->where('ident', '=', $this->loginUtilisateur)->execute()->fetchObject();
        if($secure !== false && $secure !== null && $secure->error > 2) {
            $date = new \DateTime($secure->edate);
            if($date->add(new \DateInterval('PT'.self::BLOCK_DELAY.'M')) > new \DateTime()) {
                throw new UserBlockedError('User blocked. Last attempt : '. $secure->edate);
            } else {
                $this->clearAttempt();
            }
        }
    }

    protected function attemptFailed()
    {
        if(static::$db->select()->from('cms_secure')->where('ident','=',$this->loginUtilisateur)->execute()->fetchObject() === false) {
            if(!static::$db->exec('INSERT INTO cms_secure SET error = 1, edate = NOW(), ident = "'.$this->loginUtilisateur.'"')) {
                throw new InsertError('Unable to save the row : (ident => '.$this->loginUtilisateur. ', error => 1, edate => '.date('Y-m-d H:i:s').')');
            }
        } else {
            if(!static::$db->exec('UPDATE cms_secure SET error = error +1, edate = NOW() WHERE ident = "'.$this->loginUtilisateur.'"')) {
                throw new UpdateError('Unable to update the row (ident => '.$this->loginUtilisateur. ', error => error +1, edate => '.date('Y-m-d H:i:s').')');
            }
        }
    }

    private function clearAttempt()
    {
        static::$db->delete()->from('cms_secure')->where('ident','=',$this->loginUtilisateur)->execute();
    }

    public function cleanBuild()
    {
        parent::cleanBuild();
        unset($this->id);
        $this->id_utilisateur = (int) $this->id_utilisateur;
    }


}