<?php

use Bloom\Extractor\CrmExtractor;
use Bloom\Extractor\Model\ExtractLead;
use Bloom\Extractor\Model\VehiculeInteret;
use Carbon\Carbon;
use GuzzleHttp\Client;


class Extractable implements \Bloom\Extractor\Interfaces\ICrmExtractable
{

    protected $called = [];

    public function assertCalledWith($name, $value, $strict = true)
    {
        return in_array($value, $this->called[ $name ], $strict);
    }

    public function assertCalledWithType($name, $type)
    {
        if (!array_key_exists($name, $this->called)) {
            return false;
        }
        $function = 'is_' . $type;

        return call_user_func_array($function, [$this->called[ $name ][0]]);
    }

    /**
     * Récupère un lead extractacle pour le CRM
     * La fonction peut être nulle si on accède aux données depuis la classe
     * Appelle la fonction getProvenance()
     *
     * @param mixed|null $datas
     *
     * @return array
     */
    public function getClient($datas = null)
    {
        $this->called['getClient'][] = $datas;
        $faker = \Faker\Factory::create('fr_FR');
        static $data;
        if ($data !== null) return $data;

        return $data = array_merge([
            'dateFiche'      => Carbon::now(),
            'sousProvenance' => $faker->randomElement(['SITE_JAPAUTO', 'HONDA_FRANCE']),
            'support'        => '',
            'urlInscription' => $faker->url,
            'civilite'       => $faker->title,
            'nom'            => $faker->name,
            'prenom'         => $faker->firstName,
            'codePostal'     => str_replace(' ', '', $faker->postcode),
            'ville'          => $faker->city,
            'telDomicile'    => $faker->phoneNumber,
            'telPortable'    => $faker->phoneNumber,
            'emailPerso'     => $faker->safeEmail,
            'leadId'         => $faker->numberBetween(),
            'concession'     => $faker->randomElement(['MURAT', 'PARIS11', 'COURBEVOIE']),
            'message'        => $faker->text(),
            'address'        => $faker->address,
        ], $datas ?: []);
    }

    /**
     * Récupère les données métiers propres aux véhicules.
     * Appelle l fonction getStatus()
     *
     * @param mixed|null $datas
     *
     * @return array
     */
    public function getVehiculeInteret($datas = null)
    {
        $this->called['getVehiculeInteret'][] = $datas;

        return [
            'statut'        => 'VN',
            'marque'        => 'HONDA',
            'modele'        => 'Civic',
            'phase'         => 'Ph1',
            'generation'    => 'X',
            'carrosserie'   => 'Berline 5 portes',
            'finition'      => null,
            'energie'       => null,
            'boite'         => null,
            'libeleVersion' => null,
        ];
    }

    /**
     * Récupère les données métiers propres au véhicule du demandeur.
     *
     * @param mixed|null $datas
     *
     * @return array
     */
    public function getVehiculeReprise($datas = null)
    {
        $this->called['getVehiculeReprise'][] = $datas;

        return null;
    }

    /**
     * Donne le type de provenance à la fonction getClient()
     * Doit correspondre à l'un des éléments de la constante SOUS_PROVENANCES
     *
     * @return string
     */
    public function getProvenance()
    {
        $this->called['getProvenance'][] = null;

        return 'SITE_JAPAUTO';
    }

    /**
     * Donne le type de provenance à la fonction getVehiculeInteret()
     * Doit correspondre à l'un des éléments de la constante STATUTS
     *
     * @return string
     */
    public function getStatus()
    {
        $this->called['getStatus'][] = null;

        return 'VN';
    }

    /**
     * Exceute les modifications après extraction du lead
     *
     * @param integer $leadCrmId
     *
     * @return mixed
     */
    public function afterExtract($leadCrmId)
    {
        $this->called['afterExtract'][] = $leadCrmId;

        return;
    }
}

/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 06/11/19
 * Time: 17:27
 */
class CrmExtractorTest extends PHPUnit_Framework_TestCase
{

    protected function setUp()
    {
        CrmExtractor::switchCrm('japauto');
    }

    public function getLeadData()
    {
        $faker = \Faker\Factory::create('fr_FR');

        return [
            'dateFiche'       => Carbon::now(),
            'sousProvenance'  => $faker->randomElement(['SITE_JAPAUTO', 'HONDA_FRANCE']),
            'support'         => '',
            'urlInscription'  => $faker->url,
            'civilite'        => $faker->title,
            'nom'             => $faker->name,
            'prenom'          => $faker->firstName,
            'codePostal'      => str_replace(' ', '', $faker->postcode),
            'ville'           => $faker->city,
            'telDomicile'     => $faker->phoneNumber,
            'telPortable'     => $faker->phoneNumber,
            'emailPerso'      => $faker->safeEmail,
            'leadId'          => $faker->numberBetween(),
            'concession'      => $faker->randomElement(['MURAT', 'PARIS11', 'COURBEVOIE']),
            'message'         => $faker->text(),
            'address'         => $faker->address,
            'vehiculeInteret' => new VehiculeInteret([
                'statut'        => 'VN',
                'marque'        => 'HONDA',
                'modele'        => 'Civic',
                'phase'         => 'Ph1',
                'generation'    => 'X',
                'carrosserie'   => 'Berline 5 portes',
                'finition'      => null,
                'energie'       => null,
                'boite'         => null,
                'libeleVersion' => null,
            ])
        ];
    }

    public function provider()
    {
        return [[new ExtractLead($this->getLeadData())]];
    }

    /**
     * @dataProvider provider
     */
    public function testServicesActiv($data)
    {
        $fullConf = require CONF_PATH . '/chapat_crm.php';
        $conf = $fullConf['crm'][ $fullConf['default'] ];
        $client = new Client();
        $response = $client->post($conf['url'], [
            'headers' => ['Content-Type' => 'application/json', 'Accept' => 'application/json'],
            'body'    => json_encode($data)
        ]);
        $response = json_decode($response->getBody()->getContents());

        static::assertAttributeEquals('OK', 'code', $response);
    }

    public function testParse()
    {
        $lead = new Extractable();
        $extract = CrmExtractor::parse($lead);

        static::assertInstanceOf(ExtractLead::class, $extract);
        static::assertTrue($lead->assertCalledWith('getClient', null));
        static::assertTrue($lead->assertCalledWith('getVehiculeInteret', null));

        foreach ($lead->getClient() as $name => $value) {
            if ($value instanceof Carbon) {
                if ($name === 'dateFiche') {
                    $value = $value->format("d/m/Y H:i");
                } else {
                    $value = $value->format("d/m/Y");
                }
            }
            static::assertEquals($value, $extract->$name, "Expected $name to be transposed");
        }
        foreach ($lead->getVehiculeInteret() as $name => $value) {
            if ($value instanceof Carbon) {
                $value = $value->format("d/m/Y");
            }
            static::assertEquals($value, $extract->vehiculeInteret->$name, "Expected $name to be transposed");
        }
    }


    /**
     * @dataProvider provider
     */
    public function testSend($data)
    {
        $actual = CrmExtractor::send($data);

        static::assertInstanceOf(StdClass::class, $actual);
    }

    /**
     *
     * @depends testParse
     * @depends testSend
     */
    public function testExtract()
    {
        $lead = new Extractable();

        static::assertTrue(CrmExtractor::extract($lead));
        static::assertTrue($lead->assertCalledWithType('afterExtract', 'int'));

        return [[$lead]];
    }

    public function testSwitchCrm()
    {
        static::assertTrue(CrmExtractor::switchCrm('kia'));
        $fullConf = require CONF_PATH . '/chapat_crm.php';
        static::assertEquals($fullConf['crm']['kia']['url'], CrmExtractor::getConf()['url']);
    }

    public function testExtractKia()
    {
        $lead = new Extractable();
        CrmExtractor::switchCrm('kia');
        static::assertTrue(CrmExtractor::extract($lead));
        static::assertTrue($lead->assertCalledWithType('afterExtract', 'int'));
    }

    /**
     * @depends      testExtract
     * @dataProvider testExtract
     */
    public function testExtractDoublons(Extractable $lead)
    {
        static::assertTrue(CrmExtractor::extract($lead));
        static::assertTrue($lead->assertCalledWith('afterExtract', 0));
    }

    /**
     * @expectedException \Bloom\Extractor\Exceptions\SendExtractedLeadException
     * @dataProvider provider
     */
    public function testServicesError(ExtractLead $lead)
    {
        $lead->dateFiche = null;
        $lead->codePostal = '31 310983';
        CrmExtractor::send($lead);
    }
}
