<?php
/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 06/11/19
 * Time: 15:37
 */

namespace Bloom\Extractor\Api;


use Bloom\Extractor\Exceptions\ConfigurationError;
use Bloom\Extractor\Exceptions\DuplicateLeadException;
use Bloom\Extractor\Exceptions\ExtractLeadException;
use Bloom\Extractor\Exceptions\MalFormatedResponse;
use Bloom\Extractor\Exceptions\SendExtractedLeadException;
use Bloom\Extractor\Interfaces\ICrmExtractable;
use Bloom\Extractor\Logger\Log;
use Bloom\Extractor\Model\ExtractLead;
use Bloom\Extractor\Model\VehiculeInteret;
use Bloom\Extractor\Model\VehiculeReprise;
use Carbon\Carbon;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Facades\Storage;
use phpDocumentor\Reflection\Types\Boolean;

class CrmApi
{

    /**
     * @var array
     */
    protected $conf;
    protected $fullconf;

    public function __construct(array $conf)
    {
        $this->setConf($conf);
    }

    public function switchCrm($name)
    {
        if (array_key_exists($name, $this->fullconf['crm'])) {
            $this->conf = $this->fullconf['crm'][ $name ];

            return true;
        }

        return false;
    }

    public function setConf(array $conf)
    {
        $this->fullconf = $conf;
        $conf = $conf['crm'][ $conf['default'] ];
        if (!array_key_exists('url', $conf)) {
            throw new \InvalidArgumentException('Configuration must contain the url');
        }
        $this->conf = $conf;
    }

    /**
     * @param ICrmExtractable $data
     *
     * @return bool
     * @throws ConfigurationError
     * @throws ExtractLeadException
     * @throws MalFormatedResponse
     * @throws SendExtractedLeadException
     */
    public function extract(ICrmExtractable $data)
    {
        $lead = $this->parse($data);
        try {
            $response = $this->send($lead);
            $data->afterExtract($response->leadCrmId);
        } catch (DuplicateleadException $e) {
            $data->afterExtract(0);

            return true;
        }

        return true;
    }

    /**
     * @param ICrmExtractable $data
     *
     * @return ExtractLead
     * @throws ExtractLeadException
     */
    public function parse(ICrmExtractable $data)
    {
        Log::debug(
            "parsing lead",
            [
                'data'            => $data,
                'client'          => $data->getClient(),
                'vehiculeInteret' => $data->getVehiculeInteret(),
                "vehiculeReprise" => $data->getVehiculeReprise()
            ]
        );
        try {
            $lead = new ExtractLead($data->getClient());

            if (!empty($data->getVehiculeInteret())) {
                $lead->vehiculeInteret = new VehiculeInteret($data->getVehiculeInteret());
            }

            if (!empty($data->getVehiculeReprise())) {
                $lead->vehiculeReprise = new VehiculeReprise($data->getVehiculeReprise());
            }

            return $lead;
        } catch (ExtractLeadException $e) {

            // Nested error ... of the same type
            // FIXME: Faut clean les execptions levée par les constructeurs
            throw new ExtractLeadException(
                "Error during lead parsing, error : " . $e->getMessage(),
                $e->getCode(),
                $e
            );
        }
    }

    /**
     * @param ExtractLead $lead
     *
     * @return mixed
     * @throws DuplicateLeadException
     * @throws MalFormatedResponse
     * @throws SendExtractedLeadException
     * @throws ExtractLeadException
     */
    public function send(ExtractLead $lead)
    {
        $data = json_encode($lead);
        if(!empty($data->modele) && $data->vehiculeInteret->statut == 'VO'){
            $data->modele = substr($data->modele, 0, 50);
        }
        Log::debug("Send Leads to crm", ['lead' => $data]);
        $filename = $this->fullconf['default'] . '-' . Carbon::now()->format('YmdHisuz') . '.json';
        if (defined('LARAVEL_START')) {
            // Sur les instance de laravel on stock aussi les json qui est envoyé préfixé par le crm déstinataire
            Storage::disk()
                ->put($filename, $data);
        }

        try {
            $client = new Client();
            $response = $client->post($this->conf['url'], [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept'       => 'application/json',
                    'Api-Key'      => $this->conf['key']
                ],
                'body'    => $data
            ]);
            $response = (object)json_decode($response->getBody()->getContents());
        } catch (ClientException $e) {
            $response = (object)json_decode($e->getResponse()->getBody()->getContents());
            if ($e->getResponse()->getStatusCode() == 422) {
                Log::debug("Duplicate entry JAA CRM", ['response' => $response]);
                if (defined('LARAVEL_START')) {
                    // On considéré les cas dupliqué comme étant ok, donc on supprime le fichier envoyé
                    Storage::disk()->delete($filename);
                }
                throw new DuplicateLeadException("Duplicate entry");
            } else {
                if (defined('LARAVEL_START')) {
                    Storage::disk()
                        ->put('crm-response-' . $e->getResponse()->getStatusCode() . '-' . Carbon::now()->format('YmdHis') . '.txt',
                            $e->getResponse()->getBody()->getContents()
                        );
                }
            }
            Log::debug("Send Leads to crm", ['response' => $response]);
        } catch (\Exception $e) {
            Log::debug("Bad result send CRM", ['response' => $e->getMessage()]);
            // Un truc c'est mal passé on peux pas continué en dessous
            throw new ExtractLeadException($e->getMessage(), $e->getCode(), $e);
        }

        Log::debug("Response", ['response' => $response]);

        if ($response === null) {
            throw new MalFormatedResponse(sprintf('(%s) No json response, given : %s', $this->conf['url'], $response));
        }

        if ($response->code == 'OK') {
            if (defined('LARAVEL_START')) {
                // Si tout c'est bien passé on supprime le fichier, parcequ'il faut pas déconné non plus
                Storage::disk()->delete($filename);
            }

            return $response->result;
        }
        throw new SendExtractedLeadException($response, $response->message);
    }

    public function getConf()
    {
        return $this->conf;
    }
}