<?php

namespace Tests\Unit;

use Bloom\Cms\UserModule\{Data\Autorisation, Data\Groupe, Data\User, Notifications\PasswordReset};
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Tests\Test;
use Illuminate\Foundation\Testing\RefreshDatabase;


class UtilisateurTest extends Test
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function get_list_users_without_permission()
    {
        Gate::define('UNKNOWN_PERMISSION', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $this->actingAs($user)
            ->get(route('admin_user_list'));
        $this->assertEquals('Vous n\'avez pas les droits pour voir la liste des utilisateurs', session('error'));
    }

    /**
     * @test
     */
    public function post_new_user_save_control_redirection()
    {
        Gate::define('USERS_CREATE', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $post['name'] = Str::random();
        $post['email'] = "jm.gautier@bloom-multimedia.fr";
        $response = $this->actingAs($user)
            ->post(route('admin_user_new_save'), $post);
        $response->assertRedirect(route('admin_user_list'));
    }

    /**
     * @test
     */
    public function post_new_user_save_and_send_email()
    {
        Gate::define('USERS_CREATE', function () {
            return true;
        });
        Notification::fake();
        $user = factory(User::class)->create();
        $post['name'] = "Moi";
        $post['email'] = "jm.gautier@bloom-multimedia.fr";

        $response = $this->actingAs($user)
            ->post(route('admin_user_new_save'), $post);
        $expect = User::where('email', '=', $post['email'])->first();
        $response->assertRedirect(route('admin_user_list'));
        Notification::assertSentTo($expect, PasswordReset::class);
    }

    /**
     * @test
     */
    public function post_new_user_save_control_database()
    {
        Gate::define('USERS_CREATE', function () {
            return true;
        });

        $user = factory(User::class)->create();
        $post['name'] = "Moi";
        $post['email'] = "jm.gautier@bloom-multimedia.fr";

        $this->actingAs($user)
            ->post(route('admin_user_new_save'), $post);
        $this->assertDatabaseHas('users', [
            'name'          => $post['name'],
            'email'         => $post['email'],
            'password_save' => 'non'
        ]);
    }

    /**
     * @test
     */
    public function post_new_user_already_existing()
    {
        Gate::define('USERS_CREATE', function () {
            return true;
        });

        $user = factory(User::class)->create(['name' => 'Moi', 'email' => 'jm.gautier@bloom-multimedia.fr']);
        $post['name'] = "Moi";
        $post['email'] = "jm.gautier@bloom-multimedia.fr";

        $this->actingAs($user)
            ->post(route('admin_user_new_save'), $post);
        $this->assertEquals('Le mail sélectionné existe déjà en base de données', session('error'));

    }

    /**
     * @test
     */
    public function delete_user_without_permission()
    {
        Gate::define('UNKNOWN_PERMISSION', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $this->actingAs($user)
            ->get('admin/user/1/delete');
        $this->assertEquals('Vous n\'avez pas les droits pour supprimer un utilisateur', session('error'));
    }

    /**
     * @test
     */
    public function delete_user_do_no_exist()
    {
        Gate::define('USERS_DELETE', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $this->actingAs($user)
            ->get('admin/user/2/delete');
        $this->assertEquals('L\'utilisateur n\'existe pas en base de données', session('error'));
    }

    /**
     * @test
     */
    public function delete_user_exist()
    {
        Gate::define('USERS_DELETE', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $this->actingAs($user)
            ->get('admin/user/1/delete');
        $this->assertDatabaseMissing('users', [
            'name'  => $user->name,
            'email' => $user->email
        ]);
    }

    /**
     * @test
     */
    public function edit_user_without_permission()
    {
        Gate::define('UNKNOWN_PERMISSION', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $this->actingAs($user)
            ->get('admin/user/1/edit');
        $this->assertEquals('Vous n\'avez pas les droits pour éditer un utilisateur', session('error'));
    }

    /**
     * @test
     */
    public function edit_user_do_no_exist()
    {
        Gate::define('USERS_EDIT', function () {
            return true;
        });
        $user = factory(User::class)->create(['name' => Str::random(), 'email' => 'jm.gautier@bloom-multimedia.fr']);
        $this->actingAs($user)
            ->get(route('admin_user_edit', rand(2, 9)));
        $this->assertEquals('L\'utilisateur n\'existe pas en base de données', session('error'));
    }

    /**
     * @test
     */
    public function edit_user_save_without_permission()
    {
        Gate::define('UNKNOWN_PERMISSION', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $this->actingAs($user)
            ->post(route('admin_user_edit_save', (1)));
        $this->assertEquals('Vous n\'avez pas les droits pour éditer un utilisateur', session('error'));
    }

    /**
     * @test
     */
    public function edit_user_save_control_database()
    {
        Gate::define('USERS_EDIT', function () {
            return true;
        });
        $user = factory(User::class)->create(['name' => 'Moi', 'email' => 'jm.gautier@bloom-multimedia.fr']);
        $post['name'] = 'Moi';
        $post['email'] = 'jm2.gautier@bloom-multimedia.fr';
        $this->actingAs($user)
            ->post('admin/user/1/edit/save', $post);
        $this->assertDatabaseHas('users', [
            'name'  => $post['name'],
            'email' => $post['email']
        ]);
        $this->assertEquals('Enregistrement ok', session('message'));
    }

    /**
     * @test
     */
    public function edit_user_save_no_exist()
    {
        Gate::define('USERS_EDIT', function () {
            return true;
        });
        $user = factory(User::class)->create(['name' => Str::random(), 'email' => 'jm.gautier@bloom-multimedia.fr']);
        $post['name'] = Str::random();
        $post['email'] = 'jm2.gautier@bloom-multimedia.fr';
        $this->actingAs($user)
            ->post(route('admin_user_edit_save', (rand(2, 9))), $post);
        $this->assertDatabaseMissing('users', [
            'name'  => $post['name'],
            'email' => $post['email']
        ]);
        $this->assertEquals('L\'utilisateur n\'existe pas en base de données', session('error'));
    }

    /**
     * @test
     */
    public function edit_user_save_new_group_authorization()
    {
        Gate::define('USERS_EDIT', function () {
            return true;
        });
        $user = factory(User::class)->create(['name' => Str::random(), 'email' => 'jm.gautier@bloom-multimedia.fr']);
        $nom = Str::random();
        /**
         * @var Collection|Autorisation[] $auths
         */
        $auths = factory(Autorisation::class, 4)->create();
        /**
         * @var Groupe $groupe
         */
        $groupe = factory(Groupe::class)->create();
        foreach ($auths as $auth) {
            $groupe->auths()->create([
                'groupes_id'       => $groupe->id,
                'autorisations_id' => $auth->id,
            ]);
        }
        $expect = clone $auths;
        $expect->shift();
        $post = array();
        foreach ($expect as $autorisation) {
            $post[ $autorisation->nom ] = 'on';
        }
        $post['nom'] = $nom;

        $post['name'] = Str::random();
        $post['email'] = 'jm2.gautier@bloom-multimedia.fr';
        $this->actingAs($user)
            ->post(route('admin_user_edit_save', (rand(2, 9))), $post);
        foreach ($expect as $autorisation) {
            $this->assertDatabaseHas('groupe_autorisations', [
                'groupes_id'       => $groupe->id,
                'autorisations_id' => $autorisation->id
            ]);
        }

    }
}