<?php

namespace Bloom\Cms\UserModule\Http\Controllers;

use Bloom\Cms\UserModule\{CmsUserModuleProvider, Data\Autorisation, Data\Groupe, Data\GroupeAutorisation};
use Illuminate\Http\Request;
use Illuminate\Support\Facades\{Auth, DB, Gate, Log, Session, Validator};

class GroupeController extends Controller
{
    private function admin($autorisation): bool
    {
        if (Auth::user()->super_admin) {
            return true;
        }

        return Gate::allows($autorisation);
    }

    public function index()
    {
        if (!$this->admin(CmsUserModuleProvider::USERS_G_LIST)) {
            Session::flash('error', 'Vous n\'avez pas les droits pour voir la liste des groupes');

            return back();
        }
        $groupes = Groupe::all();

        return view('User::groupe.liste_groupe', compact('groupes'));
    }

    public function nouveauGroupeSave(Request $request)
    {

        if ($this->admin('USERS_G_CREATE') == true) {
            $validator = Validator::make($request->all(), [
                'nom' => 'required|max:60',
            ]);
            if ($validator->fails()) {
                return back()
                    ->withInput(request()->all())
                    ->withErrors($validator);
            } else {
                $groupe = new Groupe();
                $nom = request('nom');
                $verifGroupe = Groupe::where('nom', $nom)->count();
                if ($verifGroupe < 1) {
                    $groupe->nom = request('nom');
                    DB::beginTransaction();
                    try {
                        if ($groupe->save()) {
                            $autorisations = Autorisation::all();
                            $groupe_autorisation = new GroupeAutorisation();
                            foreach ($autorisations as $autorisation) {
                                $autorisationNom = request($autorisation->nom);
                                if ($autorisationNom == 'on') {
                                    $groupe_autorisation->create([
                                        'groupes_id'       => $groupe->id,
                                        'autorisations_id' => $autorisation->id,
                                    ]);
                                }
                            }
                            DB::commit();
                            Session::flash('message', $groupe->nom . ' est bien enregistré');

                        } else {
                            Session::flash('error', 'Une erreur est survenu lors de l\'enregistrement');
                        }
                    } catch (\Exception $e) {
                        DB::rollBack();
                        Session::flash('error', 'Une erreur est survenu lors de l\'enregistrement');
                    }
                } else {
                    Session::flash('error', 'Le groupe sélectionné existe déjà en base de données');
                }

                return redirect(route('admin_group_list'));
            }
        } else {
            Session::flash('error', 'Vous n\'avez pas les droits pour créer un nouveau groupe');

            return back();
        }
    }

    public function edit(Groupe $groupe)
    {
        if (!$groupe->exists) {
            // Création
            if (!$this->admin(CmsUserModuleProvider::USERS_G_CREATE)) {
                Session::flash('error', 'Vous n\'avez pas les droits pour créer un nouveau groupe');

                return back();
            }
            $autorisations = Autorisation::all();

            return view('User::groupe.groupe', compact('autorisations'));
        }
        if (!$this->admin(CmsUserModuleProvider::USERS_G_EDIT)) {
            Session::flash('error', 'Vous n\'avez pas les droits pour éditer les groupes');

            return back();
        }
        $autorisations = Autorisation::all();
        $groupe_autorisations = $groupe->auths()->pluck('autorisations_id');

        return view('User::groupe.groupe', compact('groupe', 'autorisations', 'groupe_autorisations'));
    }

    public function save(Groupe $groupe, Request $request)
    {
        if ($this->admin(CmsUserModuleProvider::USERS_G_CREATE) ||
            $this->admin(CmsUserModuleProvider::USERS_G_EDIT)) {
            $validator = Validator::make($request->all(), [
                'nom' => 'required|max:60',
            ]);
            if ($validator->fails()) {
                return back()
                    ->withInput(request()->all())
                    ->withErrors($validator);
            } else {
                DB::beginTransaction();
                $autorisations = Autorisation::all();
                $groupe->auths()->detach();

                foreach ($autorisations as $autorisation) {
                    $autorisationNom = request($autorisation->nom);

                    if ($autorisationNom == 'on') {
                        $groupe->auths()->attach([
                            'autorisations_id' => $autorisation->id,
                        ]);
                    }
                }
                $groupe->nom = request('nom');
                if ($groupe->save()) {
                    DB::commit();
                    session()->flash('message', 'Enregistrement ok');
                } else {
                    session()->flash('error', "Problème d'enregistrement");
                    DB::rollBack();
                }

                return redirect(route('admin_user_group_index'));
            }
        } else {
            Session::flash('error', 'Vous n\'avez pas les droits pour créer ou édtier un groupe');

            return back();
        }

    }

    public function deleteGroupe($groupe)
    {
        /**
         * @var Groupe $groupes
         */

        $groupes = Groupe::find($groupe);
        if ($groupes->super_admin == true) {
            Session::flash('error', 'Le groupe admin ne peut être effacer');
            Log::error('error', ['Le groupe admin a voulu être effacé']);

            return redirect(route('admin_group_list'));
        }

        if ($this->admin('USERS_G_DELETE') == true) {
            if (!$groupes->delete()) {
                Session::flash('error', 'Une erreur est survenue lors de la suppression.');
            } else {
                Session::flash('message', 'Suppression réussie.');
            }

            return redirect(route('admin_user_group_index'));
        } else {
            Session::flash('error', 'Vous n\'avez pas les droits pour supprimer un groupe');

            return back();
        }

    }

}

