<?php

namespace Tests\Unit;

use Bloom\CMS\Core\Test;
use Bloom\Cms\UserModule\{CmsUserModuleProvider, Data\Autorisation, Data\Groupe, Data\User};
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Str;

class GroupeTest extends Test
{
    protected $provider = CmsUserModuleProvider::class;

    /**
     * @test
     */
    public function get_control_list_group()
    {
        Gate::define('USERS_G_LIST', function () {
            return true;
        });
        $user = factory(User::class)->create();

        $response = $this->actingAs($user)
            ->get(route('admin_user_group_index'));
        $response->assertSuccessful();
    }

    /**
     * @test
     */
    public function post_group_should_save_name()
    {
        Gate::define('USERS_G_CREATE', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $nom = Str::random();
        $response = $this->actingAs($user)
            ->post(route('admin_user_group_save'), ['nom' => $nom]);

        $this->assertStringContainsString('Enregistrement ok', session('message'));

        $response->assertRedirect(route('admin_user_group_index'));
        $this->assertDatabaseHas('groupes', [
            'nom' => $nom
        ]);
    }

    /**
     * @test
     */
    public function post_group_need_permission()
    {
        Gate::define('UNKNOWN_PERMISSION', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $nom = Str::random();
        $this->actingAs($user)
            ->post(route('admin_user_group_save'), ['nom' => $nom]);
        $this->assertEquals('Vous n\'avez pas les droits pour créer ou édtier un groupe', session('error'));
    }

    /**
     * @test
     */
    public function post_edit_group_should_save_name()
    {
        Gate::define('USERS_G_EDIT', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $nom = Str::random();
        $groupe = factory(Groupe::class)->create();
        $this->actingAs($user)
            ->post(route('admin_user_group_save'), ['groupe' => $groupe, 'nom' => $nom]);
        $this->assertStringContainsString('Enregistrement ok', session('message'));
    }

    /**
     * @test
     */
    public function post_edit_group_control_delete_auth()
    {
        Gate::define('USERS_G_EDIT', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $nom = Str::random();

        /**
         * @var Collection|Autorisation[] $auths
         */
        $auths = factory(Autorisation::class, 4)->create();

        /**
         * @var Groupe $groupe
         */
        $groupe = factory(Groupe::class)->create();
        foreach ($auths as $auth) {
            $groupe->auths()->attach($auth);
        }

        $expect = clone $auths;
        $expect->shift();

        $post = [
            'nom' => $groupe->nom,
        ];
        foreach ($expect as $autorisation) {
            $post[ $autorisation->nom ] = 'on';
        }

        $response = $this->actingAs($user)
            ->post(route('admin_user_group_save', $groupe), $post);

        $this->assertTrue($response->isRedirection());

        foreach ($expect as $autorisation) {
            $this->assertDatabaseHas('groupe_autorisations', [
                'groupes_id'       => $groupe->id,
                'autorisations_id' => $autorisation->id
            ]);
        }
        $this->assertDatabaseMissing('groupe_autorisations', [
            'groupes_id'       => $groupe->id,
            'autorisations_id' => $auths[0]->id
        ]);
    }

    /**
     * @test
     */
    public function get_delete_group()
    {
        Gate::define('USERS_G_DELETE', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $groupe = factory(Groupe::class)->create();
        $response = $this->actingAs($user)
            ->get(route('admin_user_group_delete',['groupe' => $groupe]));

        $this->assertTrue($response->isRedirection());
        $this->assertStringContainsString('Suppression réussie.', session('message'));
        $this->assertDatabaseMissing('groupes', [
            'nom' => $groupe->nom,
        ]);
    }
}
