<?php

namespace Tests\Unit;

use Bloom\CMS\Core\Test;
use Bloom\Cms\UserModule\{CmsUserModuleProvider, Data\Autorisation, Data\Groupe, Data\User};
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Str;

class GroupeTest extends Test
{
    protected $provider = CmsUserModuleProvider::class;

    /**
     * @test
     */
    public function get_control_list_group()
    {
        Gate::define('USERS_G_LIST', function () {
            return true;
        });
        $user = factory(User::class)->create();

        $response = $this->actingAs($user)
            ->get(route('admin_group_list'));
        $response->assertSuccessful();
    }

    /**
     * @test
     */
    public function post_group_should_save_name()
    {
        Gate::define('USERS_G_CREATE', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $nom = Str::random();
        $response = $this->actingAs($user)
            ->post(route('admin_group_new_save'), ['nom' => $nom]);

        $this->assertContains($nom, session('message'));

        $response->assertRedirect(route('admin_group_list'));
        $this->assertDatabaseHas('groupes', [
            'nom' => $nom
        ]);
    }

    /**
     * @test
     */
    public function post_group_need_permission()
    {
        Gate::define('UNKNOWN_PERMISSION', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $nom = Str::random();
        $this->actingAs($user)
            ->post(route('admin_group_new_save'), ['nom' => $nom]);
        $this->assertEquals('Vous n\'avez pas les droits pour créer un nouveau groupe', session('error'));
    }

    /**
     * @test
     */
    public function post_edit_group_should_save_name()
    {
        Gate::define('USERS_G_EDIT', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $nom = Str::random();
        $groupe = factory(Groupe::class)->create();
        $this->actingAs($user)
            ->post('admin/group/1/edit/save', ['groupe' => $groupe, 'nom' => $nom]);
        $this->assertContains('Enregistrement ok', session('message'));
    }

    /**
     * @test
     */
    public function post_edit_group_control_delete_auth()
    {
        Gate::define('USERS_G_EDIT', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $nom = Str::random();
        /**
         * @var Collection|Autorisation[] $auths
         */
        $auths = factory(Autorisation::class, 4)->create();
        /**
         * @var Groupe $groupe
         */
        $groupe = factory(Groupe::class)->create();
        foreach ($auths as $auth) {
            $groupe->auths()->create([
                'groupes_id'       => $groupe->id,
                'autorisations_id' => $auth->id,
            ]);
        }

        $expect = clone $auths;
        $expect->shift();
        $autorisationNom = array();
        foreach ($expect as $autorisation) {
            $autorisationNom[ $autorisation->nom ] = 'on';
        }
        $autorisationNom['nom'] = $nom;


        $response = $this->actingAs($user)
            ->post('admin/group/1/edit/save', $autorisationNom);

        $this->assertTrue($response->isRedirection());

        foreach ($expect as $autorisation) {
            $this->assertDatabaseHas('groupe_autorisations', [
                'groupes_id'       => $groupe->id,
                'autorisations_id' => $autorisation->id
            ]);
        }
        $this->assertDatabaseMissing('groupe_autorisations', [
            'groupes_id'       => $groupe->id,
            'autorisations_id' => $auths[0]->id
        ]);
    }

    /**
     * @test
     */
    public function get_delete_group()
    {
        Gate::define('USERS_G_DELETE', function () {
            return true;
        });
        $user = factory(User::class)->create();
        $groupe = factory(Groupe::class)->create();
        $response = $this->actingAs($user)
            ->get('admin/group/1/delete');
        $this->assertTrue($response->isRedirection());

        $this->assertDatabaseMissing('groupes', [
            'nom' => $groupe->nom,
        ]);
    }
}
