<?php

namespace Bloom\Cms\UserModule\Http\Controllers;

use Bloom\Cms\Sites\Models\Site;
use Bloom\Cms\Sites\Repository\Eloquent\SiteRepository;
use Bloom\Cms\UserModule\CmsUserModuleProvider;
use Bloom\Cms\UserModule\Data\Groupe;
use Bloom\Cms\UserModule\Data\User;
use Illuminate\Foundation\Auth\SendsPasswordResetEmails;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

/**
 * Class UtilisateursController
 * @package Bloom\Cms\UserModule\Http\Controllers
 */
class UtilisateursController extends Controller
{
    use SendsPasswordResetEmails;

    public function index()
    {
        if (!CmsUserModuleProvider::admin(CmsUserModuleProvider::USERS_LIST, 'Vous n\'avez pas les droits pour voir la liste des utilisateurs')) {
            return back();
        }

        $user = auth()->user();
        if ($user->super_admin) {
            $users = User::all();
        } else {
            $users = User::where('boss_id', $user->id)->get();
    }

            return view('User::identification.liste_utilisateur', compact('users'));
    }

    public function delete(User $user)
    {
        if (CmsUserModuleProvider::admin(CmsUserModuleProvider::USERS_DELETE)) {
            if ($user->super_admin == true) {
                Session::flash('error', 'Le super admin ne peut être effacer');
                Log::error('error', ['Le super admin à essayer  voulu être effacé']);

                return redirect(route('admin_user_index'));
            }
            if (!$user->exists) {
                Session::flash('error', 'L\'utilisateur n\'existe pas en base de données');
                Log::error('error', ['L\'utilisateur n\'existe pas en base de données']);

                return redirect(route('admin_user_index'));
            } elseif (!$user->delete()) {
                Session::flash('error', 'Une erreur est survenue lors de la suppression.');
                Log::error('error', ['Une erreur est survenu lors de la suppression']);

                return redirect(route('admin_user_index'));
            } else {
                Session::flash('message', 'Suppression réussie.');

                return redirect(route('admin_user_index'));
            }
        } else {
            Session::flash('error', 'Vous n\'avez pas les droits pour supprimer un utilisateur');

            return back();
        }
    }

    public function edit(User $user, SiteRepository $siteRepository)
    {
        $sites = $site_users = null;
        if (config('sites.multi_sites')) {
            $sites = $siteRepository->getEnLigne();
            $site_users = $user->sites()->get();
        }
        if (!$user->exists) {
            // création
            if (!CmsUserModuleProvider::admin(CmsUserModuleProvider::USERS_CREATE)) {
                Session::flash('error', 'Vous n\'avez pas les droits pour créer un nouvel utilisateur');

                return back();
            }
            $groupes = Groupe::all();

            return view('User::identification.utilisateur', compact('groupes', 'sites'));
        }
        if (!CmsUserModuleProvider::admin(CmsUserModuleProvider::USERS_EDIT)) {
            Session::flash('error', 'Vous n\'avez pas les droits pour éditer un utilisateur');

            return back();
        }

        $groupes = Groupe::all();
        $user_groupes = $user->groupes()->pluck("groupes_id");

        return view('User::identification.utilisateur', compact('user', 'groupes', 'user_groupes', 'sites', 'site_users'));
    }

    public function save(User $user, Request $request, SiteRepository $siteRepository)
    {
        if ($user->exists) {
            if (!CmsUserModuleProvider::admin(CmsUserModuleProvider::USERS_EDIT)) {
                Session::flash('error', 'Vous n\'avez pas les droits pour éditer un utilisateur');

                return back();
            }
            if ($user->super_admin) {
                Session::flash('error', 'Le super admin ne peut être modifié');
                Log::error('error', ['Le super admin à voulu être modifié']);

                return redirect(route('admin_user_index'));
            }
            $validator = Validator::make($request->all(), [
                'name' => 'required|max:60',
                'email' => 'required|email:rfc,dns',
                'password' => 'nullable|min:8|regex:/^(?=.*?[a-zA-Z0-9])(?=.*?[#?!@$%^&*-]).{8,}$/',
                'password_confirmation' => 'same:password',
            ]);
            if ($validator->fails()) {
                return back()
                    ->withInput(request()->all())
                    ->withErrors($validator);
            } else {
                DB::beginTransaction();
                $user->groupes()->detach();
                $groupes = Groupe::all();
                $groupesInput = $request->input('groupes');
                foreach ($groupes as $groupe) {
                    if (!is_null($groupesInput) && array_key_exists($groupe->nom, $groupesInput)) {
                        $user->groupes()->attach([
                            'groupes_id' => $groupe->id,
                        ]);
                    }
                }
                $user->sites()->detach();
                $sites = $siteRepository->getEnLigne();
                foreach ($sites as $site) {
                    $site_user = $request->input('site' . $site->id);
                    if ($site_user == 'on') {
                        $user->sites()->attach([
                            'sites_id' => $site->id,
                        ]);
                    }
                }
                $user->email = request('email');
                $user->name = request('name');
                if ($user->save()) {
                    DB::commit();
                    session()->flash('message', 'Enregistrement ok');
                } else {
                    DB::rollBack();
                    session()->flash('error', "Problème d'enregistrement");
                    Log::error('error', ['Une erreur est survenu lors de la modification de l\'utilisateur']);
                }
                return redirect(route('admin_user_index'));
            }
        } else {
            if (!CmsUserModuleProvider::admin(CmsUserModuleProvider::USERS_CREATE)) {
                Session::flash('error', 'Vous n\'avez pas les droits pour créer un nouvel utilisateur');

                return back();
            }
            $validator = Validator::make($request->all(), [
                'name' => 'required|max:60',
                'email' => 'required|email:rfc,dns',
                'password' => 'required|min:8|regex:/^(?=.*?[a-zA-Z0-9])(?=.*?[#?!@$%^&*-]).{8,}$/',
                'password_confirmation' => 'required|same:password',
            ]);
            if ($validator->fails()) {
                return back()
                    ->withInput(request()->all())
                    ->withErrors($validator);
            } else {
                $email = request('email');
                $password = request('password');

                $verifUser = User::where('email', $email)->count();
                if ($verifUser < 1) {
                    try {
                        $connectedUser = auth()->user();
                        $user->name = request('name');
                        $user->email = request('email');
                        $user->password = Hash::make($password);
                        if (!$connectedUser->super_admin && is_null($connectedUser->boss_id)) {
                            $user->boss_id = $connectedUser->id;
                        }
                        $user->groupe_site_id = $connectedUser->groupe_site_id;

                        DB::beginTransaction();
                        if ($user->save()) {
                            $groupes = Groupe::all();
                            $groupesInput = $request->input('groupes');
                            foreach ($groupes as $groupe) {
                                if (!is_null($groupesInput) && array_key_exists($groupe->nom, $groupesInput)) {
                                    $user->groupes()->attach([
                                        'groupes_id' => $groupe->id,
                                    ]);
                                }
                            }
                            $sites = $siteRepository->getEnLigne();
                            foreach ($sites as $site) {
                                $site_user = $request->input('site' . $site->id);
                                if ($site_user == 'on') {
                                    $user->sites()->attach([
                                        'sites_id' => $site->id,
                                    ]);
                                }
                            }
                            $user->save();

                            DB::commit();
                            Session::flash('message', $user->name . ' est bien enregistré');
                        } else {
                            DB::rollBack();
                            Session::flash('error', 'Une erreur est survenu lors de l\'enregistrement');

                            return back()->withInput(request()->all());
                        }
                    } catch (\Exception $e) {
                        Session::flash('error', 'Une erreur est survenu lors de l\'enregistrement');
                        Log::error('error', ['Une erreur est survenu lors de l\'enregistrement']);

                        return back()->withInput(request()->all());
                    }
                } else {
                    Session::flash('error', 'Le mail sélectionné existe déjà en base de données');

                    return back()->withInput(request()->all());
                }

//        $this->sendResetLinkEmail($request);

                return redirect(route('admin_user_index'));
            }
        }
    }

    public function edit_password(User $user)
    {
        if (!CmsUserModuleProvider::admin(CmsUserModuleProvider::USERS_EDIT)) {
            Session::flash('error', 'Vous n\'avez pas les droits pour éditer un utilisateur');

            return back();
        }

        return view('User::identification.password', compact('user'));
    }

    public function save_password(User $user, Request $request)
    {
        if (!CmsUserModuleProvider::admin(CmsUserModuleProvider::USERS_EDIT)) {
            Session::flash('error', 'Vous n\'avez pas les droits pour éditer un utilisateur');

            return back();
        }
        if ($user->super_admin) {
            Session::flash('error', 'Le super admin ne peut être modifié');
            Log::error('error', ['Le super admin à voulu être modifié']);

            return redirect(route('admin_user_index'));
        }
        $validator = Validator::make($request->all(), [
            'password' => 'nullable|min:8|regex:/^(?=.*?[a-zA-Z0-9])(?=.*?[#?!@$%^&*-]).{8,}$/',
            'password_confirmation' => 'same:password',
        ]);
        if ($validator->fails()) {
            return back()
                ->withInput(request()->all())
                ->withErrors($validator);
        } else {
            DB::beginTransaction();
            $password = request('password');
            if ($password) {
                $user->password = Hash::make($password);
            }
            if ($user->save()) {
                DB::commit();
                session()->flash('message', 'Enregistrement ok');

                return redirect(route('admin_user_index'));
            } else {
                DB::rollBack();
                session()->flash('error', "Problème d'enregistrement");
                Log::error('error', ['Une erreur est survenu lors de la modification de l\'utilisateur']);

                return redirect(route('admin_user_index'));
            }
        }
    }
}
