<?php

namespace Bloom\CMS\HPV\Specialites\Http\Controllers;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Actions;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\HPV\Specialites\Model\Specialite;
use Bloom\CMS\HPV\Specialites\Model\SpecialitesCategorie;
use Bloom\CMS\HPV\Specialites\SpecialitesServiceProvider;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiées',
                    'href'   => route('admin_specialite_index'),
                    'number' => Specialite::published()->count(),
                    'active' => request()->route()->getName() === 'admin_specialite_index',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_specialite_encours', ['orderBy' =>'page.updated_at', 'order' => 'desc']),
                    'number' => Specialite::status(Statut::EN_COURS)->count(),
                    'active' => request()->route()->getName() === 'admin_specialite_encours',
                ],
                [
                    'title'  => 'Archivées',
                    'href'   => route('admin_specialite_archived', ['orderBy' =>'page.updated_at', 'order' => 'desc']),
                    'number' => Specialite::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_specialite_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(string $publie, string $action): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_specialite_edit')
            ->addAction($publie, $action, 'bloomicon-admin-arrow-up-right')
            ->addNewTabAction('Prévisualiser', 'admin_specialite_preview');
        $sub_actions = new Actions('sub_actions');
        $sub_actions
            ->addAction('Dupliquer', 'admin_specialite_duplicate', 'bloomicon-admin-copy')
            ->addAction('Archiver', 'admin_specialite_archive', 'bloomicon-admin-archive');

        return array_merge($actions->toArray(), $sub_actions->toArray());
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');

        if (request('orderBy') == null) {
            //@codeCoverageIgnoreStart
            // La fonction est correctement appelée par la collection mais n'est pas lue par le coverage report, du coup
            // on l'ignore pour pas qu'il nous dise que la fonction est pas couverte
            $sort = function (Specialite $specialite1, Specialite $specialite2) {
                if ($specialite1->categorie_name > $specialite2->categorie_name){
                    return 1;
                }
                if ($specialite1->categorie_name < $specialite2->categorie_name) {
                    return -1;
                }
                if ($specialite1->categorie_name == $specialite2->categorie_name) {
                    if ($specialite1->nom > $specialite2->nom) {
                        return 1;
                    }
                    if ($specialite1->nom < $specialite2->nom) {
                        return -1;
                    }
                    if ($specialite1->nom == $specialite2->nom) {
                        return 0;
                    }
                }
            };
            //@codeCoverageIgnoreEnd

            return [
                'specialites' => $collection->sort($sort),
                'order'    => [
                    'item'  => request('orderBy', 'categorie_name'),
                    'order' => request('order', 'asc'),
                ]
            ];
        } else {
            //@codeCoverageIgnoreStart
            // La fonction est correctement appelé par la collection mais n'est pas lu par le coverage report, du coup
            // on l'ignore pour pas qu'il nous disent que la fonction est pas couverte
            $sort = function (Specialite $specialite) {
                return Arr::get(Arr::dot($specialite->toArray()), request('orderBy', 'nom'));
            };
            //@codeCoverageIgnoreEnd

            return [
                'specialites' => request('order', 'asc') === "asc" ?
                    $collection->sortBy($sort) :
                    $collection->sortByDesc($sort),
                'order'    => [
                    'item'  => request('orderBy', 'page.updated_at'),
                    'order' => request('order', 'desc'),
                ]
            ];
        }

    }

    public function index()
    {
        $actions = $this->getDefaultActions('Dépublier', 'admin_specialite_unpublish');

        $data = array_merge($this->getSorted(Specialite::published()->get()), $actions);

        return view('Specialite::admin.index', $this->getViewData($data));
    }

    public function encours()
    {
        $actions = $this->getDefaultActions('Publier', 'admin_specialite_publish');

        $data = array_merge($this->getSorted(Specialite::status(Statut::EN_COURS)->get()), $actions);

        return view('Specialite::admin.index', $this->getViewData($data));
    }

    public function archived()
    {
        $actions = new Actions();
        $actions
            ->addNewTabAction('Prévisualiser', 'admin_specialite_preview')
            ->addAction('Désarchiver', 'admin_specialite_unpublish');

        $data = array_merge($this->getSorted(Specialite::archived()->get()), $actions->toArray());

        return view('Specialite::admin.index', $this->getViewData($data));
    }

    public function edit(Specialite $specialite)
    {
        $categories = SpecialitesCategorie::all();

        return view('Specialite::admin.edit', compact('specialite', 'categories'));
    }

    public function save(Specialite $specialite)
    {
        $statut = request('statut_id', Statut::EN_COURS);
        $specialite->nom = request('nom');
        $specialite->specialites_categorie_id = request('specialites_categorie_id');
        $specialite->contenu = request('autres_infos', '');
        $specialite_id = $specialite->id;
        $specialite->save();

        if (!$specialite_id) {
            $page = new Page();
            $page->contenu()->associate($specialite);
            $page->multiplicity = "1";
            $page->meta_titre = "";
            $page->meta_description = "";
            $page->dossier()->associate(Dossier::query()->where('name', '=', SpecialitesServiceProvider::DOSSIER_NAME)->first());
            $page->created_by = Auth::id();
        } else {
            $page = $specialite->page;
        }

        $page->titre = request('nom');
        $page->slug = request('slug');
        $page->intro = request('introduction', '');
        $page->mentions = request('mentions_legales', '');
        $page->image_opengraph = request('photo', '');
        $page->to_publish_at = request('date_pub');
        $page->to_unpublish_at = request('date_depub');
        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = 2;
        }
        if ($statut == 1) {
            $page->to_publish_at = now();
        }
        $page->statut_id = $statut;
        $page->last_modified_by = Auth::id();
        $page->save();

        if ((int)$specialite->page->statut_id === Statut::PUBLIE) {
            return redirect()->route('admin_specialite_index');
        }

        return redirect()->route('admin_specialite_encours');
    }

    public function preview(Specialite $specialite)
    {
        return $specialite->getHandler()->single($specialite);
    }

    public function duplicate(Specialite $specialite)
    {
        $page = $specialite->page;
        $page2 = $page->replicate();
        $page2->titre .= ' - Copie';
        $page2->slug .= '-copie';
        $page2->pathname .= '-copie';
        $page2->save();

        $specialite2 = $specialite->replicate();
        $specialite2->nom .= ' - Copie';
        $specialite2->save();
        $page2->contenu()->associate($specialite2);
        $page2->save();

        return redirect()->back();
    }

    public function publish(Specialite $specialite)
    {
        $specialite->page->statut_id = Statut::PUBLIE;
        $specialite->page->to_publish_at = now();
        $specialite->page->save();

        return redirect()->back();
    }

    public function unpublish(Specialite $specialite)
    {
        $specialite->page->statut_id = Statut::EN_COURS;
        $specialite->page->to_unpublish_at = now();
        $specialite->page->save();

        return redirect()->back();
    }

    public function archive(Specialite $specialite)
    {
        $specialite->page->statut_id = Statut::ARCHIVE;
        $specialite->page->to_unpublish_at = now();
        $specialite->page->save();

        return redirect()->back();
    }

    public function delete(Specialite $specialite)
    {
        $specialite->page->statut_id = Statut::SUPPRIMER;
        $specialite->page->save();

        return redirect()->back();
    }
}
