<?php

/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Recrutement\Http\Controllers;

use Bloom\CMS\Core\Helpers\Actions;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\HPV\Cliniques\Model\Clinique;
use Bloom\CMS\Modules\Recrutement\Model\Recrutement;
use Bloom\CMS\Modules\Recrutement\Model\Service;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;

/**
 * Class AdminController
 *
 * @package Bloom\CMS\Modules\Recrutement\Http\Controllers
 */
class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiés',
                    'href'   => route('admin_recrutement_index'),
                    'number' => Recrutement::published()->count(),
                    'active' => request()->route()->getName() === 'admin_recrutement_index',
                ],
                [
                    'title'  => 'Archivés',
                    'href'   => route('admin_recrutement_archived'),
                    'number' => Recrutement::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_recrutement_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_recrutement_edit')
            ->addNewTabAction('Prévisualiser', 'admin_recrutement_preview')
            ->addAction('Archiver', 'admin_recrutement_archive');

        return array_merge($actions->toArray(), isset($sub_actions) ? $sub_actions->toArray() : []);
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');
        //@codeCoverageIgnoreStart
        // La fonction est correctement appelé par la collection mais n'est pas lu par le coverage report, du coup
        // on l'ignore pour pas qu'il nous disent que la fonction est pas couverte
        $sort = function (Recrutement $recrutement) {
            return Arr::get(Arr::dot($recrutement->toArray()), request('orderBy', 'id'));
        };

        //@codeCoverageIgnoreEnd

        return [
            'recrutements' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order'    => [
                'item'  => request('orderBy', 'id'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions();

        // On ajoute une action devant l'action de base


        $data = array_merge($this->getSorted(Recrutement::published()->get()), $actions);

        return view('Recrutement::admin.listing', $this->getViewData($data));
    }


    public function archived()
    {
        $actions = $this->getDefaultActions();

        // On vire les valeurs 0 et 2
        unset($actions['actions'][0]);
        unset($actions['actions'][2]);

        $data = array_merge($this->getSorted(Recrutement::archived()->get()), $actions);

        return view('Recrutement::admin.listing', $this->getViewData($data));
    }

    public function edit(Recrutement $recrutement)
    {
        $cliniques = Clinique::all();
        $services = Service::all();//A modifier quand la relation sera créé sur le module clinique
        $dossiers = Dossier::all();
        $page = $recrutement->page;
        if (request('dossier_id')) {
            if ($page === null) {
                $page = new Page();
            }
            $page->dossier_id = request('dossier_id');
        }

        return view('Recrutement::admin.edit', compact('recrutement', 'dossiers', 'page', 'cliniques', 'services'));
    }

    /**
     * Permet de fill les parametre générique d'une page avec le formulaire
     *
     * @param Page $page
     */
    protected function fillPage(Page $page): void
    {
        $page->titre = request('titre');
        $page->intro = request('description'); // pas d'intro dans les annonces donc on reprend la description comme indiqué dans le brief
        $page->slug = request('slug');
        $page->image_opengraph = request('vignette');
        $page->mentions = request('mentions');
        $page->updatePathname();
    }

    /**
     * Crée le recrutement et sa page en valorisant correctement le statut, celui-ci pouvant être surchargé en fonction
     * des valeurs du formulaire
     *
     * @param Recrutement $recrutement
     * @param int     $statut
     *
     * @return int
     */
    public function createRecrutement(Recrutement $recrutement, int $statut): int
    {
        // On crée le recrutement
        $recrutement->contenu = request('description');
        $recrutement->save();
        $recrutement->reference .= $recrutement->id;
        $recrutement->save();

        /**
         * @var Dossier $dossier
         */
        $dossier = Dossier::where('name', '=', 'Annonces')->first();

        // On crée la page
        $page = new Page();
        $page->meta_titre = '';
        $page->meta_description = '';
        $page->created_by = Auth::user()->getAuthIdentifier();
        $page->last_modified_by = Auth::user()->getAuthIdentifier();
        $page->multiplicity = Page::SINGLE;
        $page->dossier_id = $dossier->id; //Toutes les annonces sont dans le dossier Annonces pour gérer les redirections.
        $this->fillPage($page);
        $page->to_publish_at = now();
        $page->statut_id = $statut;



        // On attache la page et le recrutement
        $page->contenu()->associate($recrutement);
        $page->save();

        return $statut;
    }

    /**
     * Modifie le recrutement et sa page en valorisant correctement le statut, celui-ci pouvant être surchargé en fonction
     * des valeurs du formulaires
     *
     * @param Recrutement $recrutement
     * @param int     $statut
     *
     * @return int
     */
    public function editRecrutement(Recrutement $recrutement, int $statut): int
    {
        /**
         * @var Page $page
         */
        $page = $recrutement->page;

        $recrutement->contenu = request('description');

        if ($recrutement->isDirty()) {
            // On mets à jour la page
            $page->last_modified_by = Auth::user()->getAuthIdentifier();
            $recrutement->save();
            $page->touch();
        }

        $this->fillPage($page);

        $page->statut_id = $statut;

        if ($page->isDirty()) {
            $page->last_modified_by = Auth::user()->getAuthIdentifier();
            $page->save();
            $recrutement->touch();
        }

        return $statut;
    }

    public function save(Recrutement $recrutement)
    {
        $recrutement->clinique_id = request('lieu');
        $recrutement->service_id = request('service');
        $recrutement->contrat_type = request('contrat_type');
        $recrutement->date_embauche = request('date-embauche');
        $recrutement->duree = request('duree');

        // Création de la référence de l'annonce
        if (!$recrutement->exists) {
            $clinique = Clinique::findOrFail(request('lieu'));
            $service = Service::findOrFail(request('service'));
            $recrutement->reference = '#hpv' . substr(strtolower($clinique->nom),0,3) . substr(strtolower($service->nom),0,3);
        }

        $statut = 1;

        if ($recrutement->page !== null) {
            $statut = $recrutement->page->statut_id ?: 1;
        }
        $statut = $recrutement->exists ? $this->editRecrutement($recrutement, $statut) : $this->createRecrutement($recrutement, $statut);

        if ($statut == 0) {
            return redirect()->route('admin_recrutement_archived');
        }

        return redirect()->route('admin_recrutement_index');
    }

    public function preview(Recrutement $recrutement)
    {
        return $recrutement->getHandler()->single($recrutement);
    }

    public function archive(Recrutement $recrutement)
    {
        /**
         * @var Page $page
         */
        $page = $recrutement->page;
        $page->archive();

        return redirect()->back();
    }

    public function publish(Recrutement $recrutement)
    {
        /**
         * @var Page $page
         */
        $page = $recrutement->page;
        $page->publish();

        return redirect()->back();
    }
}
