<?php
/**
 * Created by : loic
 * Created at : 22/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Tests\Feature;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Recrutement\CmsRecrutementServiceProvider;
use Bloom\CMS\Modules\Recrutement\Model\Recrutement;
use Faker\Factory;
use Illuminate\Foundation\Auth\User;

/**
 * Class RecrutementTest
 * @package Tests\Feature
 */
class RecrutementTest extends \Bloom\CMS\Core\Test
{
    protected $provider = CmsRecrutementServiceProvider::class;

    public function test_index_recrutements_should_show_recrutements()
    {
        $response = $this->actingAs(User::first())->get(route('admin_recrutement_index'));

        $response->isOk();
        $response->assertViewIs('Recrutement::admin.listing');
        $response->assertViewHas('recrutements');
    }

    public function test_edit_recrutement_should_show_form()
    {
        $response = $this->actingAs(User::first())->get(route('admin_recrutement_edit'));

        $response->isOk();
        $response->assertViewIs('Recrutement::admin.edit');
    }

    public function test_edit_recrutement_should_show_form_with_recrutement()
    {
        /**
         * @var Recrutement $recrutement
         */
        $recrutement = factory(Recrutement::class)->create();
        $recrutement->page = factory(Page::class)->create();

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_edit', $recrutement));

        $response->isOk();
        $response->assertViewIs('Recrutement::admin.edit');

        /**
         * @var Recrutement $actual
         */
        $actual = $response->viewData('recrutement');

        $this->assertInstanceOf(Recrutement::class, $actual);
        $this->assertTrue($actual->exists);
    }

    public function form_provider()
    {
        $this->setUp();
        $faker = Factory::create();
        $array = [];
        for ($i = 0; $i < 10; $i++) {
            $array[] = [
                [ // Form
                    'titre'      => $faker->text(50),
                    'slug'       => $faker->slug,
                    'vignette'   => 'vignette.jpg',
                    // Les html ont un \n qui est trim par Laravel, donc on le supprime directement dans la source
                    'contenu'    => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'intro'      => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'dossier_id' => null,
                    'statut_id'  => 1
                ]
            ];
        }

        return $array;
    }

    /**
     * @param array $form
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_save_recrutement(array $form)
    {
        $response = $this->actingAs(User::first())->post(route('admin_recrutement_save'), $form);

        $response->isOk();
        $recrutement = Recrutement::first();

        $this->assertDatabaseHas('recrutements', [
            'contenu' => $form['contenu']
        ]);
        $this->assertDatabaseHas('pages', [
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'intro'           => $form['intro'],
            'image_opengraph' => $form['vignette'],
            'contenu_type'    => 'recrutements',
            'contenu_id'      => $recrutement->id
        ]);
    }


    /**
     * @param array $form
     *
     * @depends      test_post_edit_should_save_recrutement
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_edit_recrutement(array $form)
    {
        // Cela permet de crée le recrutement / page necessaire sans ce prendre la tête a tout réecrire
        $this->test_post_edit_should_save_recrutement($form);

        /**
         * @var Recrutement $recrutement
         * @var Page    $page
         */
        $recrutement = Recrutement::first();
        $page = $recrutement->page;

        // On modifie des trucs
        $form['titre'] = 'test des truc';
        $form['slug'] = 'test-des-truc';
        // On fait comme si l'image n'était pas touché
        $form['image'] = $page->image_opengraph;


        $response = $this->actingAs(User::first())->post(route('admin_recrutement_save', $recrutement), $form);
        $response->isOk();

        $this->assertDatabaseHas('recrutements', [
            'contenu' => $form['contenu']
        ]);

        $this->assertDatabaseHas('pages', [
            'id'              => $page->id,
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'image_opengraph' => $form['image']
        ]);
    }

    public function test_preview_should_a_front_page()
    {
        $recrutement = \factory(Recrutement::class)->create();
        $page = \factory(Page::class)->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_preview', $recrutement));
        $response->assertOk();
        $response->assertViewIs('Recrutement::front.index');
        static::assertInstanceOf(Page::class, $response->viewData('page'));
        static::assertEquals($page->id, $response->viewData('page')->id);
        static::assertInstanceOf(Recrutement::class, $response->viewData('recrutement'));
        static::assertEquals($recrutement->id, $response->viewData('recrutement')->id);
    }

    public function test_preview_should_a_front_page_even_if_not_published()
    {
        $recrutement = \factory(Recrutement::class)->create();
        $page = \factory(Page::class)->state('unpublished')->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_preview', $recrutement));
        $response->assertOk();
        $response->assertViewIs('Recrutement::front.index');
        static::assertInstanceOf(Page::class, $response->viewData('page'));
        static::assertEquals($page->id, $response->viewData('page')->id);
        static::assertInstanceOf(Recrutement::class, $response->viewData('recrutement'));
        static::assertEquals($recrutement->id, $response->viewData('recrutement')->id);
    }

    public function test_preview_should_401_if_not_connected()
    {
        $recrutement = \factory(Recrutement::class)->create();
        $page = \factory(Page::class)->state('unpublished')->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);
        // On force le content type attendu sur du json pour pas être redirigé vers le login, et avoir le bon statut
        $response = $this->get(route('admin_recrutement_preview', $recrutement), ['Accept' => 'application/json']);
        $response->assertUnauthorized();
    }

    public function test_programmed_should_filtered_corectly()
    {
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)->state('unpublished')->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_programmes'));
        $response->assertOk();
        $response->assertViewIs('Recrutement::admin.listing');
        $recrutements = $response->viewData('recrutements');
        static::assertEquals(0, $recrutement->count());

        $page->to_publish_at = now()->addWeek();
        $page->save();

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_programmes'));
        $response->assertOk();
        $response->assertViewIs('Recrutement::admin.listing');
        $recrutements = $response->viewData('recrutements');
        static::assertEquals(1, $recrutements->count());
    }

    public function test_encours_should_filtred_out_the_programmed_one()
    {
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)->state('unpublished')->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_encours'));
        $response->assertOk();
        $response->assertViewIs('Recrutement::admin.listing');
        $recrutements = $response->viewData('recrutements');
        static::assertEquals(1, $recrutements->count());

        $page->to_publish_at = now()->addWeek();
        $page->save();

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_encours'));
        $response->assertOk();
        $response->assertViewIs('Recrutement::admin.listing');
        $recrutements = $response->viewData('recrutements');
        static::assertEquals(0, $recrutements->count());
    }

    public function test_archived_should_only_show_archived_one()
    {
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        \factory(Page::class)
            ->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id, 'statut_id' => 0]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_archived'));
        $response->assertOk();
        $response->assertViewIs('Recrutement::admin.listing');
        $recrutements = $response->viewData('recrutements');
        static::assertEquals(1, $recrutements->count());

        // On ajoute un recrutement publié pour voir
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        \factory(Page::class)
            ->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_archived'));
        $response->assertOk();
        $response->assertViewIs('Recrutement::admin.listing');
        $recrutements = $response->viewData('recrutements');
        static::assertEquals(1, $recrutements->count());
    }

    public function test_shouldnt_show_deleted_recrutement()
    {
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        \factory(Page::class)
            ->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id, 'statut_id' => Statut::SUPPRIMER]);

        foreach (['admin_recrutement_archived', 'admin_recrutement_encours', 'admin_recrutement_programmes', 'admin_recrutement_index'] as $route) {
            $response = $this->actingAs(User::first())->get(route($route));
            $response->assertOk();
            $response->assertViewIs('Recrutement::admin.listing');
            $recrutements = $response->viewData('recrutements');
            static::assertEquals(0, $recrutements->count());
        }
    }

    public function test_action_archive_should_archied_and_redirect()
    {
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)
            ->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_archive', $recrutement));
        $response->assertRedirect();

        $page->refresh();
        static::assertEquals(Statut::ARCHIVE, $page->statut_id);
    }

    public function test_action_publish_should_publish_and_redirect()
    {
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)
            ->state('unpublished')
            ->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_publish', $recrutement));
        $response->assertRedirect();

        $page->refresh();
        static::assertEquals(Statut::PUBLIE, $page->statut_id);
        static::assertNotEmpty($page->to_publish_at);
    }

    public function test_action_unpublish_should_unpublish_and_redirect()
    {
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)
            ->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_unpublish', $recrutement));
        $response->assertRedirect();

        $page->refresh();
        static::assertEquals(Statut::EN_COURS, $page->statut_id);
        static::assertEmpty($page->to_publish_at);
    }

    public function test_action_delete_should_softdelete_and_redirect()
    {
        $recrutement = \factory(Recrutement::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)
            ->create(['contenu_type' => 'recrutements', 'contenu_id' => $recrutement->id]);

        $response = $this->actingAs(User::first())->get(route('admin_recrutement_delete', $recrutement));
        $response->assertRedirect();

        $page->refresh();
        static::assertEquals(Statut::SUPPRIMER, $page->statut_id);
    }
}