<?php
/**
 * Created by : loic
 * Created at : 22/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Tests\Feature;


use Bloom\CMS\Modules\Architecture\Model\Page;
use Bloom\CMS\Modules\Produits\Model\Produit;
use Faker\Factory;
use Illuminate\Foundation\Auth\User;
use Illuminate\Http\Testing\File;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\Test;

class ProduitTest extends Test
{

    public function setUp(): void
    {
        parent::setUp();
        Storage::fake();
    }

    public function test_index_produits_should_show_produits()
    {
        $response = $this->actingAs(new User())->get(route('admin_prevoyance_index'));

        $response->isOk();
        $response->assertViewIs('Produits::admin.listing');
        $response->assertViewHas('produits');
    }

    public function test_edit_produit_should_show_form()
    {
        $response = $this->actingAs(new User())->get(route('admin_prevoyance_edit'));

        $response->isOk();
        $response->assertViewIs('Produits::admin.edit');
    }

    public function test_edit_produit_should_show_form_with_produit()
    {
        /**
         * @var Produit $produit
         */
        $produit = factory(Produit::class)->create();
        $produit->page = factory(Page::class)->create();

        $response = $this->actingAs(new User())->get(route('admin_prevoyance_edit', $produit));

        $response->isOk();
        $response->assertViewIs('Produits::admin.edit');

        /**
         * @var Produit $actual
         */
        $actual = $response->viewData('produit');

        $this->assertInstanceOf(Produit::class, $actual);
        $this->assertTrue($actual->exists);
    }

    public function form_provider()
    {
        $this->setUp();
        $faker = Factory::create();
        $array = [];
        for ($i = 0; $i < 10; $i++) {
            $array[] = [
                [ // Form
                    'titre'      => $faker->text(50),
                    'slug'       => $faker->slug,
                    'vignette'      => UploadedFile::fake()->image('vignette.jpg'),
                    // Les html ont un \n qui est trim par Laravel, donc on le supprime directement dans la source
                    'contenu'    => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'intro'      => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'dossier_id' => null,
                    'statut_id'  => 1
                ]
            ];
        }

        return $array;
    }

    /**
     * @param array $form
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_save_produit(array $form)
    {

        $user = new User();
        $user->id = 1;
        $response = $this->actingAs($user)->post(route('admin_prevoyance_save'), $form);

        $response->isOk();

        $this->assertDatabaseHas('produits', [
            'contenu' => $form['contenu']
        ]);
        $this->assertDatabaseHas('pages', [
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'intro'           => $form['intro'],
            'image_opengraph' => $form['vignette']
        ]);

        $produit = Produit::first();

        $this->assertDatabaseHas('page_contenus', [
            'contenu_id' => $produit->id,
            'page_id'    => $produit->page->id
        ]);
    }


    /**
     * @param array $form
     *
     * @depends      test_post_edit_should_save_produit
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_edit_produit(array $form)
    {

        $user = new User();
        $user->id = 1;

        // Sa permet de crée l'produit / page necessaire sans ce prendre la tête a tout réecrire
        $this->test_post_edit_should_save_produit($form);

        /**
         * @var Produit $produit
         * @var Page    $page
         */
        $produit = Produit::first();
        $page = $produit->page;

        // On modifie des trucs
        $form['titre'] = 'test des truc';
        $form['slug'] = 'test-des-truc';
        // On fait comme si l'image n'était pas touché
        $form['image'] = $page->image_opengraph;


        $response = $this->actingAs($user)->post(route('admin_prevoyance_save', $produit), $form);
        $response->isOk();

        $this->assertDatabaseHas('produits', [
            'contenu' => $form['contenu']
        ]);

        $this->assertDatabaseHas('pages', [
            'id'              => $page->id,
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'image_opengraph' => $form['image']
        ]);

    }


    /**
     * @param array $form
     *
     * @depends      test_post_edit_should_save_produit
     *
     * @dataProvider form_provider
     */
    public function test_handler_should_return_the_given_view(array $form)
    {

        $user = new User();
        $user->id = 1;

        // Sa permet de crée l'produit / page necessaire sans ce prendre la tête a tout réecrire
        $this->test_post_edit_should_save_produit($form);

        /**
         * @var Produit $produit
         * @var Page    $page
         */
        $produit = Produit::first();

        $response = $this->createTestResponse($produit->getHandler()->single($produit));

        $response->isOk();
        $response->assertViewIs('Produits::front.index');
    }
}