<?php
/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Produits\Http\Controllers;


use Bloom\CMS\Modules\Architecture\Helpers\Actions;
use Bloom\CMS\Modules\Architecture\Model\Dossier;
use Bloom\CMS\Modules\Architecture\Model\Page;
use Bloom\CMS\Modules\Produits\Model\Produit;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;


class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiés',
                    'href'   => route('admin_prevoyance_index'),
                    'number' => Produit::published()->count(),
                    'active' => request()->route()->getName() === 'admin_prevoyance_index',
                ],
                [
                    'title'  => 'Programmés',
                    'href'   => route('admin_prevoyance_programmes'),
                    'number' => Produit::programmed()->count(),
                    'active' => request()->route()->getName() === 'admin_prevoyance_programmes',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_prevoyance_encours'),
                    'number' => Produit::status(2)->notProgrammed()->count(),
                    'active' => request()->route()->getName() === 'admin_prevoyance_encours',
                ],
                [
                    'title'  => 'Archivés',
                    'href'   => route('admin_prevoyance_archived'),
                    'number' => Produit::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_prevoyance_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(bool $with_sub = true): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_prevoyance_edit')
            ->addNewTabAction('Prévisualiser', 'admin_prevoyance_preview')
            ->addAction('Nouvelle version', '');
        if ($with_sub) {
            $sub_actions = new Actions('sub_actions');
            $sub_actions
                ->addAction('Dupliquer', 'admin_prevoyance_duplicate')
                ->addAction('Publier maintenant', 'admin_prevoyance_publish')
                ->addAction('Archiver', 'admin_prevoyance_archive');
        }

        return array_merge($actions->toArray(), isset($sub_actions) ? $sub_actions->toArray() : []);
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');
        $sort = function (Produit $produit) {
            return Arr::get(Arr::dot($produit->toArray()), request('orderBy', 'id'));
        };

        return [
            'produits' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order'    => [
                'item'  => request('orderBy', 'id'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions();

        // On remplace la 2ieme action du menu
        $actions['sub_actions'][1] = [
            'name' => 'Dépublier', 'href' => 'admin_prevoyance_unpublish'
        ];

        $data = array_merge($this->getSorted(Produit::published()->get()), $actions);

        return view('Produits::admin.listing', $this->getViewData($data));
    }

    public function programmed()
    {
        $data = array_merge($this->getSorted(Produit::programmed()->get()), $this->getDefaultActions());

        return view('Produits::admin.listing', $this->getViewData($data));
    }

    public function encours()
    {
        $data = array_merge($this->getSorted(Produit::status(2)->notProgrammed()->get()), $this->getDefaultActions());

        return view('Produits::admin.listing', $this->getViewData($data));
    }

    public function archived()
    {
        $data = array_merge(
            $this->getSorted(Produit::archived()->get()),
            (new Actions())
                ->addAction('Désarchiver', 'admin_prevoyance_unpublish')
                ->addAction('Supprimer', 'admin_prevoyance_delete')->toArray()
        );

        return view('Produits::admin.listing', $this->getViewData($data));
    }

    public function edit(Produit $produit)
    {
        $dossiers = Dossier::all();
        $page = null;
        if ($produit->page->count()) {
            $page = $produit->page->first();
        }

        return view('Produits::admin.edit', compact('produit', 'dossiers', 'page'));
    }

    public function save(Produit $produit)
    {
        $user_id = Auth::user()->getAuthIdentifier();

        if (request()->has('statut_id')) {
            $statut = request('statut_id');
        } else {
            if ($produit->page !== null && $produit->page->count() > 0) {
                $statut = $produit->page->first()->statut_id ?: 2;
            } else {
                $statut = 2;
            }
        }

        if (!$produit->exists) {
            // Nouveau

            // On crée l'produit
            $produit->contenu = request('contenu');
            $produit->save();

            // On crée la page
            $page = new Page();
            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->meta_titre = '';
            $page->meta_description = '';
            $page->created_by = $user_id;
            $page->last_modified_by = $user_id;
            $page->intro = request('intro');
            $page->statut_id = $statut;
            $page->dossier_id = request('dossier_id');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->multiplicity = Page::SINGLE;
            $image = request()->file('vignette');
            $page->image_opengraph = Storage::disk('images')->url($image->storePubliclyAs('', $image->getClientOriginalName(), 'images'));

            $page->save();

            // On attache la page et l'produit
            $produit->page()->attach($page->getKey());
        } else {
            // Edition

            /**
             * @var Page $page
             */
            $page = $produit->page->first();

            $produit->contenu = request('contenu');
            if ($produit->isDirty()) {
                // On mets à jour la page
                $page->last_modified_by = $user_id;
                $produit->save();
            }

            $page->statut_id = $statut;
            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->intro = request('intro');
            $page->dossier_id = request('dossier_id');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');

            if (request()->hasFile('vignette')) {
                $image = request()->file('vignette');
                $page->image_opengraph = Storage::disk('images')->url($image->storePubliclyAs('', $image->getClientOriginalName(), 'images'));
            }
            if ($page->isDirty()) {
                $page->last_modified_by = $user_id;
                $page->save();
                $produit->touch();
            }
        }

        if ($statut == 2) {
            if ($page->to_publish_at) {
                return redirect()->route('admin_prevoyance_programmes');
            }

            return redirect()->route('admin_prevoyance_encours');
        }

        return redirect()->route('admin_prevoyance_index');
    }

    public function preview(Produit $produit)
    {
        return $produit->getHandler()->single($produit);
    }

    public function archive(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page->first();
        $page->archive();

        return redirect()->back();
    }

    public function publish(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page->first();
        $page->publish();

        return redirect()->back();
    }

    public function unpublish(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page->first();
        $page->unpublish();

        return redirect()->back();
    }

    public function duplicate(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page->first();
        $page2 = $page->replicate();
        $page2->titre .= ' - Copie';
        $page2->save();

        $produit2 = $produit->replicate();
        $produit2->save();
        $produit2->page()->attach($page2);

        return redirect()->back();
    }

    public function delete(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page->first();
        $page->softDelete();

        return redirect()->back();
    }

}