<?php
/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Produits\Http\Controllers;

use Bloom\CMS\Core\Helpers\Actions;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Produits\Model\Produit;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;

/**
 * Class AdminController
 * @package Bloom\CMS\Modules\Produits\Http\Controllers
 */
class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiés',
                    'href'   => route('admin_prevoyance_index'),
                    'number' => Produit::published()->count(),
                    'active' => request()->route()->getName() === 'admin_prevoyance_index',
                ],
                [
                    'title' => 'Programmés' ,
                    'href' => route('admin_prevoyance_programmes', ['orderBy' => 'to_publish_at', 'order' => 'desc']),
                    'number' => Produit::programmed()->count(),
                    'active' => request()->route()->getName() === 'admin_prevoyance_programmes',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_prevoyance_encours', ['orderBy' =>'page.updated_at', 'order' => 'desc']),
                    'number' => Produit::status(2)->notProgrammed()->count(),
                    'active' => request()->route()->getName() === 'admin_prevoyance_encours',
                ],
                [
                    'title'  => 'Archivés',
                    'href'   => route('admin_prevoyance_archived', ['orderBy' =>'page.updated_at', 'order' => 'desc']),
                    'number' => Produit::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_prevoyance_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(bool $with_sub = true): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_prevoyance_edit')
            ->addNewTabAction('Prévisualiser', 'admin_prevoyance_preview')
            ->addAction('Publier', 'admin_prevoyance_publish', 'bloomicon-admin-arrow-up-right');
        if ($with_sub) {
            $sub_actions = new Actions('sub_actions');
            $sub_actions
                ->addAction('Archiver', 'admin_prevoyance_archive', 'bloomicon-admin-archive');
        }

        return array_merge($actions->toArray(), isset($sub_actions) ? $sub_actions->toArray() : []);
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');
        $sort = function (Produit $produit) {
            return Arr::get(Arr::dot($produit->toArray()), request('orderBy', 'id'));
        };

        return [
            'produits' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order'    => [
                'item'  => request('orderBy', 'id'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions();

        // On remplace la 2ieme action du menu
        $actions['actions'][2] = [
            'name' => 'Dépublier', 'href' => 'admin_prevoyance_unpublish'
        ];

        $data = array_merge($this->getSorted(Produit::published()->get()), $actions);

        return view('Prevoyance::admin.listing', $this->getViewData($data));
    }

    public function programmed()
    {
        $data = array_merge($this->getSorted(Produit::programmed()->get()), $this->getDefaultActions());

        return view('Prevoyance::admin.listing', $this->getViewData($data));
    }

    public function encours()
    {
        $data = array_merge($this->getSorted(Produit::status(2)->notProgrammed()->get()), $this->getDefaultActions());

        return view('Prevoyance::admin.listing', $this->getViewData($data));
    }

    public function archived()
    {
        $data = array_merge(
            $this->getSorted(Produit::archived()->get()),
            (new Actions())
                ->addAction('Désarchiver', 'admin_prevoyance_unpublish')
                ->addAction('Supprimer', 'admin_prevoyance_delete')->toArray()
        );

        return view('Prevoyance::admin.listing', $this->getViewData($data));
    }

    public function sortDossiers($dossiers, int $level): \Illuminate\Support\Collection
    {
        $tab = collect();

        foreach ($dossiers as $dossier) {
            $dossier->level = $level;
            $tab[] = $dossier;
            if ($dossier->childs) {
                $tab = $tab->concat($this->sortDossiers($dossier->childs, $level + 1));
            }
        }
        return $tab;
    }

    public function edit(Produit $produit)
    {
        $tableDossiers = Dossier::query()->whereNull('parent_id')->get();

        $page = $produit->page?:new Page();
        if (request('dossier_id')) {
            $page->dossier_id = request('dossier_id');
        }

        $dossiers = $this->sortDossiers($tableDossiers, 0);

        return view('Prevoyance::admin.edit', compact('produit', 'dossiers', 'page'));
    }

    public function save(Produit $produit)
    {
        $user_id = Auth::user()->getAuthIdentifier();

        if (request('statut_id')) {
            $statut = request('statut_id');
        } else {
            if ($produit->page !== null) {
                $statut = $produit->page->statut_id ?: 2;
            } else {
                $statut = 2;
            }
        }

        if (!$produit->exists) {
            // Nouveau

            // On crée l'produit
            $produit->contenu = request('contenu');
            $produit->save();

            // On crée la page
            $page = new Page();
            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->meta_titre = '';
            $page->meta_description = '';
            $page->created_by = $user_id;
            $page->last_modified_by = $user_id;
            $page->intro = request('intro');
            $page->statut_id = $statut;
            $page->dossier_id = request('dossier_id');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->multiplicity = Page::SINGLE;
            $page->image_opengraph = request('vignette');
            $page->mentions = request('mentions');

            // On attache la page et l'produit
            $page->contenu()->associate($produit);
            $page->save();

        } else {
            // Edition

            /**
             * @var Page $page
             */
            $page = $produit->page;

            $produit->contenu = request('contenu');
            if ($produit->isDirty()) {
                // On mets à jour la page
                $page->last_modified_by = $user_id;
                $produit->save();
            }

            $page->statut_id = $statut;
            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->intro = request('intro');
            $page->dossier_id = request('dossier_id');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->image_opengraph = request('vignette');
            $page->mentions = request('mentions');

            if ($page->isDirty()) {
                $page->last_modified_by = $user_id;
                $page->save();
                $produit->touch();
            }
        }

        if ($statut == 2) {
            if ($page->to_publish_at) {
                return redirect()->route('admin_prevoyance_programmes');
            }

            return redirect()->route('admin_prevoyance_encours');
        }

        return redirect()->route('admin_prevoyance_index');
    }

    public function preview(Produit $produit)
    {
        return $produit->getHandler()->single($produit);
    }

    public function archive(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page;
        $page->archive();

        return redirect()->back();
    }

    public function publish(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page;
        $page->publish();

        return redirect()->back();
    }

    public function unpublish(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page;
        $page->unpublish();

        return redirect()->back();
    }

    public function duplicate(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page;
        $page2 = $page->replicate();
        $page2->titre .= ' - Copie';
        $page2->pathname .= '-copie';
        $page2->slug .= '-copie';
        $page2->save();

        $produit2 = $produit->replicate();
        $produit2->save();
        $page2->contenu()->associate($produit2);
        $page2->save();


        return redirect()->back();
    }

    public function delete(Produit $produit)
    {
        /**
         * @var Page $page
         */
        $page = $produit->page;
        $page->softDelete();

        return redirect()->back();
    }

}
