<?php

namespace Bloom\CMS\HPV\Praticiens\Http\Controllers;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Actions;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Bloom\CMS\HPV\Praticiens\Model\Praticien;
use Bloom\CMS\HPV\Cliniques\Model\Clinique;
use Bloom\CMS\HPV\Specialites\Model\Specialite;
use Bloom\CMS\HPV\Praticiens\PraticiensServiceProvider;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{

	public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiés',
                    'href'   => route('admin_praticien_index'),
                    'number' => Praticien::published()->count(),
                    'active' => request()->route()->getName() === 'admin_praticien_index',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_praticien_encours', ['orderBy' =>'page.updated_at', 'order' => 'desc']),
                    'number' => Praticien::status(Statut::EN_COURS)->count(),
                    'active' => request()->route()->getName() === 'admin_praticien_encours',
                ],
                [
                    'title'  => 'Archivés',
                    'href'   => route('admin_praticien_archived', ['orderBy' =>'page.updated_at', 'order' => 'desc']),
                    'number' => Praticien::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_praticien_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

	public function getDefaultActions(string $publie, string $action): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_praticien_edit')
            ->addAction($publie, $action, 'bloomicon-admin-arrow-up-right')
            ->addNewTabAction('Prévisualiser', 'admin_praticien_preview');
        $sub_actions = new Actions('sub_actions');
        $sub_actions
            ->addAction('Dupliquer', 'admin_praticien_duplicate', 'bloomicon-admin-copy')
            ->addAction('Archiver', 'admin_praticien_archive', 'bloomicon-admin-archive');

        return array_merge($actions->toArray(), $sub_actions->toArray());
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');
        //@codeCoverageIgnoreStart
        // La fonction est correctement appelé par la collection mais n'est pas lu par le coverage report, du coup
        // on l'ignore pour pas qu'il nous disent que la fonction est pas couverte
        $sort = function (Praticien $praticien) {
            return Arr::get(Arr::dot($praticien->toArray()), request('orderBy', 'nom'));
        };

        //@codeCoverageIgnoreEnd

        return [
            'praticiens' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order'    => [
                'item'  => request('orderBy', 'nom'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

	public function index()
    {
        $actions = $this->getDefaultActions('Dépublier', 'admin_praticien_unpublish');

        $data = array_merge($this->getSorted(Praticien::published()->get()), $actions);

        return view('Praticien::admin.index', $this->getViewData($data));
    }

    public function encours()
    {
        $actions = $this->getDefaultActions('Publier', 'admin_praticien_publish');

        $data = array_merge($this->getSorted(Praticien::status(Statut::EN_COURS)->get()), $actions);

        return view('Praticien::admin.index', $this->getViewData($data));
    }

    public function archived()
    {
        $actions = new Actions();
        $actions
            ->addNewTabAction('Prévisualiser', 'admin_praticien_preview')
            ->addAction('Désarchiver', 'admin_praticien_unpublish');

        $data = array_merge($this->getSorted(Praticien::archived()->get()), $actions->toArray());

        return view('Praticien::admin.index', $this->getViewData($data));
    }

    public function edit(Praticien $praticien)
    {
        $jours = [
            ["val"=>"lundi", "name"=>"Lundi"],
            ["val"=>"mardi", "name"=>"Mardi"],
            ["val"=>"mercredi", "name"=>"Mercredi"],
            ["val"=>"jeudi", "name"=>"Jeudi"],
            ["val"=>"vendredi", "name"=>"Vendredi"],
            ["val"=>"samedi", "name"=>"Samedi"],
        ];
        $specialites_all = Specialite::published()->get();
        $specialites = $praticien->specialites;

        $horaires = $praticien->horaires;
        $cliniques = Clinique::published()->get();
        return view('Praticien::admin.edit', compact('praticien', 'cliniques', 'jours', 'horaires', 'specialites', 'specialites_all'));
    }

    public function save(Praticien $praticien)
    {
        $statut = request('statut_id', Statut::EN_COURS);
        $praticien->nom = request('nom');
        $praticien->prenom = request('prenom');
        $praticien->doctolib = request('doctolib');
        $praticien->cliniques_id = request('cliniques_id');
        $praticien->telephone = request('telephone');
        $praticien->horaires = request('horaires');
        $praticien->description = request('autres_infos', '');
        $praticien->photo = request('photo', '');
        $praticien_id = $praticien->id;
        $praticien->save();

        if (!$praticien_id) {
            $page = new Page();
            $page->contenu()->associate($praticien);
            $page->multiplicity = "1";
            $page->meta_titre = "";
            $page->meta_description = "";
            $page->dossier()->associate(Dossier::query()->where('name', '=', PraticiensServiceProvider::DOSSIER_NAME)->first());
            $page->created_by = Auth::id();
        } else {
            $page = $praticien->page;
        }

        $page->titre = request('prenom')." ".request('nom');
        $page->slug = request('slug');
        $page->intro = request('introduction', '');
        $page->mentions = request('mentions_legales', '');
        $page->image_opengraph = request('vignette', '');
        $page->to_publish_at = request('date_pub');
        $page->to_unpublish_at = request('date_depub');

        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = 2;
        }
        if ($statut == 1) {
            $page->to_publish_at = now();
        }
        $page->statut_id = $statut;
        $page->last_modified_by = Auth::id();
        $page->save();

        $praticien->specialites()->sync(json_decode(request('specialites'), true));

        if ((int)$praticien->page->statut_id === Statut::PUBLIE) {
            return redirect()->route('admin_praticien_index');
        }

        return redirect()->route('admin_praticien_encours');
    }

    public function preview(Praticien $praticien)
    {
        return $praticien->getHandler()->single($praticien);
    }

    public function duplicate(Praticien $praticien)
    {
        $dub = $praticien->replicate(['created_at', 'updated_at', 'created_by', 'last_modified_by']);
        $dub->nom = $praticien->nom . ' - Copie';
        $dub->save();

        $page = new Page();
        $page->titre = $praticien->page->titre . " - Copie";
        $page->slug = $praticien->page->slug . "-copie";
        $page->meta_description = "";
        $page->intro = $praticien->page->intro;
        $page->mentions = $praticien->page->mentions;
        $page->meta_titre = "";
        $page->multiplicity = "1";
        $page->statut_id = $praticien->page->statut_id;
        $page->to_publish_at = $praticien->page->to_publish_at;
        $page->to_unpublish_at = $praticien->page->to_unpublish_at;
        $page->dossier_id = $praticien->page->dossier_id;
        $page->contenu()->associate($dub);
        $page->save();
        
        return redirect()->back();
    }

    public function publish(Praticien $praticien)
    {
        $praticien->page->statut_id = Statut::PUBLIE;
        $praticien->page->to_publish_at = now();
        $praticien->page->save();

        return redirect()->back();
    }

    public function unpublish(Praticien $praticien)
    {
        $praticien->page->statut_id = Statut::EN_COURS;
        $praticien->page->to_unpublish_at = now();
        $praticien->page->save();

        return redirect()->back();
    }

    public function archive(Praticien $praticien)
    {
        $praticien->page->statut_id = Statut::ARCHIVE;
        $praticien->page->to_unpublish_at = now();
        $praticien->page->save();

        return redirect()->back();
    }

    public function delete(Praticien $praticien)
    {
        $praticien->page->statut_id = Statut::SUPPRIMER;
        $praticien->page->save();

        return redirect()->back();
    }
}