<?php

namespace Bloom\Cms\Offre\Models;

use Bloom\Cms\Gamme\Models\Modele;
use Bloom\Cms\Sites\Models\Site;
use Bloom\MSC\Services\GammesEtOffres\Cms;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Foundation\Auth\User;

/**
 * Class Offre
 * @package Bloom\Gamme\Models
 *
 * @property integer $modele_id
 * @property string  $titre
 * @property string  $label_prive
 * @property string  $entete
 * @property string  $part_1
 * @property string  $part_2
 * @property string  $visuel_principal
 * @property string  $visuel_mobile
 * @property string  $visuel_hp_bureau
 * @property string  $visuel_hp_mobile
 * @property string  $vignette_1
 * @property string  $vignette_2
 * @property string  $vignette_3
 * @property Carbon  $to_publish_at
 * @property Carbon  $to_unpublish_at
 * @property integer $created_by
 * @property integer $last_modified_by
 * @property integer $statut_id
 * @property string  $mentions_legales
 *
 */
class Offre extends Model
{

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'modele_id',
        'titre',
        'label_prive',
        'entete',
        'part_1',
        'part_2',
        'visuel_principal',
        'visuel_mobile',
        'visuel_hp_bureau',
        'visuel_hp_mobile',
        'vignette_1',
        'vignette_2',
        'vignette_3',
        'to_publish_at',
        'to_unpublish_at',
        'created_by',
        'last_modified_by',
        'statut_id',
        'mentions_legales',
    ];

    protected $dates = [
        'to_publish_at',
        'to_unpublish_at',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    protected $appends = [
        'visuel_principal_url',
        'visuel_mobile_url',
        'visuel_hp_bureau_url',
        'visuel_hp_mobile_url',
        'vignette_1_url',
        'vignette_2_url',
        'vignette_3_url',
    ];

    public function scopeStatus(Builder $query, int $status)
    {
        return $query->where('statut_id', '=', $status);
    }

    public function scopePublished(Builder $query)
    {
        return $this->scopeStatus($query, 1);
    }

    public function cree_par(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function derniere_modification_par(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function getVisuelPrincipalUrlAttribute(): ?string
    {
        if ($this->visuel_principal) {
            return url('/') . $this->visuel_principal;
        }

        return null;
    }

    public function getVisuelMobileUrlAttribute(): ?string
    {
        if ($this->visuel_mobile) {
            return url('/') . $this->visuel_mobile;
        }

        return null;
    }

    public function getVisuelHpBureauUrlAttribute(): ?string
    {
        if ($this->visuel_hp_bureau) {
            return url('/') . $this->visuel_hp_bureau;
        }

        return null;
    }

    public function getVisuelHpMobileUrlAttribute(): ?string
    {
        if ($this->visuel_hp_mobile) {
            return url('/') . $this->visuel_hp_mobile;
        }

        return null;
    }

    public function getVignette1UrlAttribute(): ?string
    {
        if ($this->vignette_1) {
            return url('/') . $this->vignette_1;
        }

        return null;
    }

    public function getVignette2UrlAttribute(): ?string
    {
        if ($this->vignette_2) {
            return url('/') . $this->vignette_2;
        }

        return null;
    }

    public function getVignette3UrlAttribute(): ?string
    {
        if ($this->vignette_3) {
            return url('/') . $this->vignette_3;
        }

        return null;
    }

    public function sites()
    {
        return $this->belongsToMany(Site::class, 'site_offres', 'offres_id', 'sites_id')->withPivot(
            ['created_at', 'updated_at']
        );
    }

    public function getModeleAttribute()
    {
        static $modeles;
        if ($modeles === null) {
            $modeles = collect(Cms::request('GET', 'gammes'))->sortBy('label_public');
        }

        return $modeles->where('id', $this->modele_id)->first();
    }
}
