<?php

namespace Bloom\Cms\Offre\Http\Controllers;

use App\Http\Controllers\Controller;
use Bloom\CMS\Core\Contenus\Statut;
use Bloom\Cms\Offre\Models\Offre;
use Bloom\Cms\Offre\Repository\Eloquent\OffreRepository;
use Bloom\Cms\Sites\Helpers\SiteHelper;
use Bloom\Cms\Sites\Repository\Eloquent\SiteRepository;
use Bloom\MSC\Services\GammesEtOffres\Cms;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class OffreController extends Controller
{

    public function index(OffreRepository $offreRepository, Request $request)
    {
        $published = SiteHelper::filter($offreRepository->getPublished());
        $programmed = SiteHelper::filter($offreRepository->getProgrammed());
        $encours = SiteHelper::filter($offreRepository->getEncours());
        $archived = SiteHelper::filter($offreRepository->getArchived());
        $nbPublished = count($published);
        $nbProgrammed = count($programmed);
        $nbEncours = count($encours);
        $nbArchived = count($archived);
        $type = $request->get('type') ?? 'published';
        switch ($type) {
            case 'programmed':
                $offres = $programmed;
                break;
            case 'encours':
                $offres = $encours;
                break;
            case 'archived':
                $offres = $archived;
                break;
            default:
                $offres = $published;
        }

        return view('Offre::liste_offre', compact('offres', 'nbPublished', 'nbProgrammed', 'nbEncours', 'nbArchived', 'type'));
    }

    public function edit(Offre $offre, SiteRepository $siteRepository)
    {
        $modeles = collect(json_decode(Cms::raw('GET', 'gammes'), true))->sortBy('label_public');
        $sites = $site_offres = null;
        if (config('sites.multi_sites')) {
            $sites = $siteRepository->getEnLigne();
            $site_offres = $offre->sites()->get();
        }

        return view('Offre::offre', compact('modeles', 'offre', 'sites', 'site_offres'));
    }

    public function save(Offre $offre, Request $request, SiteRepository $siteRepository)
    {
        $fields = [
            'modele' => 'required',
            'titre' => 'required|max:45',
            'label_prive' => 'required|max:45',
            'entete' => 'required',
            'visuel_principal' => 'required',
            'visuel_mobile' => 'required',
            'visuel_hp_bureau' => 'required',
            'visuel_hp_mobile' => 'required',
        ];

        $validator = Validator::make($request->all(), $fields);
        if ($validator->fails()) {
            return back()
                ->withInput(request()->all())
                ->withErrors($validator);
        } else {
            DB::beginTransaction();

            $user = auth()->user();

            $offre->modele_id = $request->get('modele');
            $offre->titre = $request->get('titre');
            $offre->label_prive = $request->get('label_prive');
            $offre->entete = $request->get('entete');
            $offre->part_1 = $request->get('part_1');
            $offre->part_2 = $request->get('part_2');
            $offre->visuel_principal = $this->save_visuel($request->get('visuel_principal'));
            $offre->visuel_mobile = $this->save_visuel($request->get('visuel_mobile'));
            $offre->visuel_hp_bureau = $this->save_visuel($request->get('visuel_hp_bureau'));
            $offre->visuel_hp_mobile = $this->save_visuel($request->get('visuel_hp_mobile'));
            $offre->vignette_1 = $this->save_visuel($request->get('vignette_1'));
            $offre->vignette_2 = $this->save_visuel($request->get('vignette_2'));
            $offre->vignette_3 = $this->save_visuel($request->get('vignette_3'));
            $offre->mentions_legales = $request->get('mentions_legales');
            if (!$offre->created_by) {
                $offre->created_by = $user->id;
            }
            $offre->last_modified_by = $user->id;
            $now = now()->format('Y-m-d');
            if ($offre->statut_id === Statut::EN_COURS && intval($request->get('statut_id')) === Statut::PUBLIE) {
                $offre->statut_id = Statut::PUBLIE;
                $offre->to_publish_at = $now;
            } else {
                $offre->to_publish_at = $request->get('to_publish_at');
            }
            $offre->to_unpublish_at = $request->get('to_unpublish_at');
            if ($offre->statut_id === Statut::PUBLIE && !is_null($offre->to_publish_at) && $offre->to_publish_at->format('Y-m-d') > $now) {
                $offre->statut_id = Statut::EN_COURS;
            }
            if ($offre->statut_id === Statut::EN_COURS && !is_null($offre->to_publish_at) && $offre->to_publish_at->format('Y-m-d') === $now) {
                $offre->statut_id = Statut::PUBLIE;
            }
            if (is_null($offre->statut_id)) {
                $offre->statut_id = intval($request->get('statut_id'));
            }

            if ($offre->save()) {
                // gestion multisites
                $offre->sites()->detach();
                $multisite = request('multisite');
                if (!empty($multisite)) {
                    foreach ($multisite as $site_id) {
                        $offre->sites()->attach([
                            'sites_id' => $site_id,
                        ]);
                    }
                }
                $offre->save();
                DB::commit();
                session()->flash('message', 'Enregistrement ok');
            } else {
                DB::rollBack();
                session()->flash('error', "Problème d'enregistrement");
                Log::error('error', ['Une erreur est survenu lors de la modification de l\'offre']);
            }
        }
        return redirect(route('admin_offre_index'));
    }

    private function save_visuel($path)
    {
        if ($path === null || $path === '/images/vignette_default.png') {
            return null;
        }

        return $path;
    }

    public function preview(Offre $offre)
    {

    }

    public function publish(Offre $offre)
    {
        $offre->statut_id = Statut::PUBLIE;
        $offre->to_publish_at = now()->format('Y-m-d');
        $offre->save();

        return redirect(route('admin_offre_index'));
    }

    public function unpublish(Offre $offre)
    {
        $offre->statut_id = Statut::EN_COURS;
        $offre->to_publish_at = null;
        $offre->to_unpublish_at = null;
        $offre->save();

        return redirect(route('admin_offre_index'));
    }

    public function archive(Offre $offre)
    {
        $offre->statut_id = Statut::ARCHIVE;
        $offre->to_publish_at = null;
        $offre->save();

        return redirect(route('admin_offre_index'));
    }

    public function unarchive(Offre $offre)
    {
        $offre->statut_id = Statut::EN_COURS;
        $offre->to_publish_at = null;
        $offre->to_unpublish_at = null;
        $offre->save();

        return redirect(route('admin_offre_index'));
    }

    public function delete(Offre $offre)
    {
        $offre->statut_id = Statut::SUPPRIMER;
        $offre->save();

        return redirect(route('admin_offre_index'));
    }

}
