<?php
/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Menu\Model;

use Bloom\CMS\Modules\Architecture\Contenus\Contenu;
use Bloom\CMS\Modules\Architecture\Model\{Form, Miroir, Page, Statut};
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\Log;


/**
 * Class Article
 * @package Bloom\CMS\Modules\Produits\Model
 *
 *
 * @property string            label
 * @property string            href
 * @property string            picto
 * @property string            couleur
 * @property string            couleur_bg
 * @property string            type
 * @property Page              page
 * @property Miroir            miroir
 * @property Statut            statut
 * @property integer           statut_id
 * @property integer           ordre
 * @property boolean           break
 *
 * @property boolean           can_have_child
 * @property Menu              parent
 * @property Menu[]|Collection child
 * @property Menu[]|Collection parents
 * @property Form              form
 *
 * @method static Builder|static published()
 * @method static Builder|static archived()
 * @method static Builder|static type(string $type)
 */
class Menu extends Model
{
    const STICKY = 'sticky';
    const MAIN = 'main';
    const FOOTER = 'footer';
    const SOCIAL = 'social';

    /**
     * The "booted" method of the model.
     *
     * @return void
     */
    protected static function booted()
    {
        static::addGlobalScope('ordre', function (Builder $builder) {
            $builder->orderBy('ordre');
        });
        static::addGlobalScope('not_deleted', function (Builder $builder) {
            $builder->where('statut_id', '!=', Statut::SUPPRIMER);
        });
    }

    public function form()
    {
        return $this->belongsTo(Form::class);
    }

    public function child()
    {
        return $this->hasMany(Menu::class, 'parent_id', 'id');
    }

    public function parent()
    {
        return $this->belongsTo(Menu::class, 'parent_id', 'id');
    }

    public function cree_par()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function deriere_modification_par()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function in()
    {
        if ($this->type !== self::STICKY) {
            Log::warning('Checking StickIn for non sticky menu', ['menu' => $this]);
        }

        return $this->hasMany(StickIn::class);
    }

    public function page()
    {
        return $this->belongsTo(Page::class);
    }

    public function miroir()
    {
        return $this->belongsTo(Miroir::class);
    }

    public function scopePublished(Builder $query)
    {
        return $this->scopeStatus($query, Statut::PUBLIE);
    }

    public function scopeArchived(Builder $query)
    {
        return $this->scopeStatus($query, Statut::ARCHIVE);
    }

    public function scopeStatus(Builder $query, int $status)
    {
        return $query->where('statut_id', '=', $status);
    }

    public function scopeType(Builder $query, string $type)
    {
        return $query->where('type', '=', $type);
    }


    public function getPageTypeAttribute()
    {
        return class_basename($this->page->contenu);
    }


    public function softDelete(): bool
    {
        $this->statut_id = Statut::SUPPRIMER;
        if (!$this->save()) {
            return false;
        }

        if ($this->child->count() > 0) {
            foreach ($this->child as $child) {
                // On soft delete en recursif les enfant
                $child->softDelete();
            }
        }

        if ($this->parent) {
            foreach ($this->parent->child as $ordre => $child) {
                $child->ordre = $ordre;
                $child->save();
            }
        } else {
            foreach (Menu::query()->whereNull('parent_id')->where('type', '=', $this->type)->get() as $k => $menu) {
                $menu->ordre = $k;
                $menu->save();
            }
        }

        return true;
    }

    public function getIsInAttribute(): bool
    {
        if ($this->type !== self::STICKY) {
            return true;
        }

        static $page;
        if ($page === null) {
            $path = request()->getPathInfo();

            /**
             * @var Page $page
             */
            $page = Page::where('pathname', '=', $path)->first();
            if ($page === null) {
                return false;
            }
        }

        foreach ($this->in as $stickIn) {
            /**
             * @var StickIn $stickIn
             */
            if ($stickIn->page_id == $page->id) {
                return true;
            }

            if ($stickIn->multiplicity === $page->multiplicity) {
                /**
                 * @var Contenu $contenu
                 */
                $contenu = $page->contenu;
                switch ($stickIn->multiplicity) {
                    case Page::SINGLE:
                    case Page::ALL:
                        if ($contenu instanceof $stickIn->contenu_type) {
                            return true;
                        }
                }
            }
        }

        return false;
    }

    public function getParentsAttribute()
    {
        $parents = collect();
        $parent = $this->parent;
        while ($parent !== null) {
            $parents[] = $parent;
            $parent = $parent->parent;
        }

        return $parents;

    }


    public function getCanHaveChildAttribute()
    {
        switch ($this->type) {
            case self::STICKY:
                // no break;
            case self::SOCIAL:
                return false;
                break;
            case self::FOOTER:
                // rubrique
                return $this->parents->count() < 1;
                break;
            case self::MAIN:
                // no break;
            default:
                // rubrique -> sous rubrique > sous-sous rubrique
                return $this->parents->count() < 3;
        }
    }

    public function getRedirectionOnAttribute()
    {
        return false;
    }

    public function getActiveAttribute()
    {
        if ($this->page) {
            return request()->getPathInfo() === $this->page->pathname;
        }

        return false;
    }

    public function getLienAttribute()
    {
        if ($this->page) {
            return $this->page->pathname;
        }

        if ($this->miroir) {
            return $this->miroir->pathname;
        }

        if ($this->href) {
            return $this->href;
        }

        if ($this->form) {
            return "javascript:document.dispatchEvent(new CustomEvent('form.open-modal', {detail: {view:'" . $this->form->view . "'}}))";
        }

        return "#";
    }

    public function getCanHaveLienAttribute()
    {
        return true;
    }

    public function getHasLienAttribute()
    {
        if (!$this->exists) {
            return true;
        }
        if ($this->miroir !== null) {
            return true;
        }
        if ($this->page !== null) {
            return true;
        }
        if ((string)$this->href !== '') {
            return true;
        }

        return false;
    }
}