<?php
/**
 * Created by : loic
 * Created at : 22/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Tests\Unit;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Test;
use Bloom\CMS\Modules\Meas\CmsMeasServiceProvider;
use Bloom\CMS\Modules\Meas\Model\Mea;
use Carbon\Carbon;
use Faker\Factory;
use Illuminate\Foundation\Auth\User;

class MeaTest extends Test
{
    protected $provider = CmsMeasServiceProvider::class;

    public function test_should_save_mea()
    {
        $user = new User();
        $user->id = 1;
        $mea = factory(Mea::class)->make()->toArray();
        $faker = Factory::create();
        $mea['statut_id'] = $faker->randomElement([Statut::PUBLIE, Statut::EN_COURS]);

        $response = $this->actingAs($user)->post(route('admin_mea_save'), $mea);

        $response->assertRedirect();

        $this->assertDatabaseHas('meas', [
            'titre' => $mea['titre'],
            'soustitre' => $mea['soustitre'],
            'mentions' => $mea['mentions'],
            'visuel_1' => $mea['visuel_1'],
            'visuel_2' => $mea['visuel_2'],
        ]);
    }

    public function test_edit_should_save_mea()
    {
        $user = new User();
        $user->id = 1;

        /**
         * @var Mea $mea
         */
        $mea = Mea::first();
        $mea_new = $mea->toArray();

        //On modifie des champs
        $faker = Factory::create();
        $mea_new['titre'] = $faker->sentence;
        $mea_new['soustitre'] = $faker->sentence;


        $response = $this->actingAs($user)->post(route('admin_mea_save', $mea), $mea_new);


        $response->assertRedirect();

        $this->assertDatabaseHas('meas', [
            'titre' => $mea_new['titre'],
        ]);

        $this->assertDatabaseMissing('meas', [
            'titre' => $mea->titre,
        ]);
    }

    public function test_unpublish_mea()
    {
        $user = new User();
        $user->id = 1;
        $mea = factory(Mea::class)->make()->toArray();
        $mea['date_pub'] = Carbon::now()->toDateString();

        $this->actingAs($user)->post(route('admin_mea_save'), $mea);

        /**
         * @var Mea $mea
         */
        $mea = Mea::published()->first();

        $response = $this->actingAs($user)->get(route('admin_mea_unpublish', $mea));

        $response->assertRedirect();

        $this->assertDatabaseHas('meas', [
            'statut_id' => Statut::EN_COURS,
        ]);

    }

    public function test_publish_mea()
    {
        $user = new User();
        $user->id = 1;
        $mea = factory(Mea::class)->make()->toArray();
        $mea['date_pub'] = Carbon::now()->addDays(5)->toDateString();

        $this->actingAs($user)->post(route('admin_mea_save'), $mea);

        /**
         * @var Mea $mea
         */
        $mea = Mea::programmed()->first();

        $response = $this->actingAs($user)->get(route('admin_mea_publish', $mea));

        $response->assertRedirect();

        $this->assertDatabaseHas('meas', [
            'statut_id' => Statut::PUBLIE,
        ]);

    }

    public function test_delete_mea()
    {
        $user = new User();
        $user->id = 1;
        $mea = factory(Mea::class)->make()->toArray();

        $this->actingAs($user)->post(route('admin_mea_save'), $mea);

        /**
         * @var Mea $mea
         */
        $mea = Mea::orderBy('id', 'desc')->first();

        $response = $this->actingAs($user)->get(route('admin_mea_delete', $mea));

        $response->assertRedirect();

        $this->assertDatabaseHas('meas', [
            'id' => $mea['id'],
            'statut_id' => 3
        ]);

    }

    public function test_reorder_mea()
    {
        $user = new User();
        $user->id = 1;
        $faker = Factory::create();
        $mea['statut_id'] = Statut::PUBLIE;

        $this->actingAs($user)->post(route('admin_mea_save'), $mea);

        $meas = Mea::published()->where('type', '=', 'AFF')->get();
        $new_meas = $meas->sortByDesc('rank');

        $ordre = [];

        foreach ($new_meas as $mea) {
            array_push($ordre, $mea->id);
        }

        $response = $this->actingAs($user)->post(route('admin_mea_reorder'), ['ordre' => $ordre]);

        $response->assertStatus(200);

        foreach ($meas as $index => $mea) {
            $this->assertDatabaseHas('meas', [
                'id' => $mea->id,
                'rank' => array_search($mea->id, $ordre),
            ]);
        }
    }
}