<?php

/**
 * Created by : Jérôme
 * Created at : 19/06/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Meas\Http\Controllers;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Actions;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Meas\Model\Mea;
use Bloom\CMS\Modules\Meas\Model\Type;
use Carbon\Carbon;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Arr;


/**
 * Class AdminController
 * @package Bloom\CMS\Modules\Meas\Http\Controllers
 */
class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        $type = $this->getTypeFromRoute(request()->route()->getName());

        if ($type == Type::VIDEO) {
            return [];
        } else {
            return [
                'sub_menu_items' => [
                    [
                        'title' => 'Publié' . ($type !== Type::PRINCIPAL ? 'es' : ''),
                        'href' => $type === Type::PRINCIPAL ? route('admin_mea_index') : route('admin_mea_' . strtolower($type)),
                        'number' => Mea::type($type)->published()->count(),
                        'active' => request()->route()->getName() === 'admin_mea_index' || request()->route()->getName() === 'admin_mea_aff' || request()->route()->getName() === 'admin_mea_video' || request()->route()->getName() === 'admin_mea_bandeaux',
                    ],
                    [
                        'title' => 'Programmé' . ($type !== Type::PRINCIPAL ? 'es' : 's'),
                        'href' => route('admin_mea_programmes', ['type' => $type, 'orderBy' => 'to_publish_at', 'order' => 'desc']),
                        'number' => Mea::type($type)->programmed()->count(),
                        'active' => request()->route()->getName() === 'admin_mea_programmes',
                    ],
                    [
                        'title' => 'En cours de rédaction',
                        'href' => route('admin_mea_encours', ['type' => $type, 'orderBy' => 'updated_at', 'order' => 'desc']),
                        'number' => Mea::type($type)->status(Statut::EN_COURS)->notProgrammed()->count(),
                        'active' => request()->route()->getName() === 'admin_mea_encours',
                    ],
                    [
                        'title' => 'Archivé' . ($type !== Type::PRINCIPAL ? 'es' : 's'),
                        'href' => route('admin_mea_archived', ['type' => $type, 'orderBy' => 'updated_at', 'order' => 'desc']),
                        'number' => Mea::type($type)->archived()->count(),
                        'active' => request()->route()->getName() === 'admin_mea_archived',
                    ],
                ]
            ];
        }
    }

    public function getViewData(array $data): array
    {

        $data['type'] = $this->getTypeFromRoute(request()->route()->getName());

        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(bool $with_sub = true): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_mea_edit')
            ->addNewTabAction('Prévisualiser', 'admin_mea_preview');
        if ($with_sub) {
            $sub_actions = new Actions('sub_actions');
            $sub_actions
                ->addAction('Dupliquer', 'admin_mea_duplicate', 'bloomicon-admin-copy');
        }

        return array_merge($actions->toArray(), isset($sub_actions) ? $sub_actions->toArray() : []);
    }

    public function getSorted(Collection $collection)
    {
        $sort = function (Mea $meas) {
            return Arr::get(Arr::dot($meas->toArray()), request('orderBy', 'rank'));
        };

        return [
            'meas' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order' => [
                'item'  => request('orderBy', 'rank'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions();

        // On ajoute des actions au sous-menu
        $actions['sub_actions'][1] = [
            'name' => 'Dépublier', 'href' => 'admin_mea_unpublish', 'picto' => 'bloomicon-admin-arrow-down-left'
        ];
        $actions['sub_actions'][2] = [
            'name' => 'Archiver', 'href' => 'admin_mea_archive', 'picto' => 'bloomicon-admin-archive'
        ];
        $data = array_merge($this->getSorted(Mea::type(Type::PRINCIPAL)->published()->get()), $actions);

        $data['type'] = Type::PRINCIPAL;
        $data['soustitre'] = 'Visuel principal';

        return view('Mea::admin.listing', $this->getViewData($data));
    }

    public function affiche()
    {
        $actions = $this->getDefaultActions();

        // On ajoute des actions au sous-menu
        $actions['sub_actions'][1] = [
            'name' => 'Dépublier', 'href' => 'admin_mea_unpublish', 'picto' => 'bloomicon-admin-arrow-down-left'
        ];
        $actions['sub_actions'][2] = [
            'name' => 'Archiver', 'href' => 'admin_mea_archive', 'picto' => 'bloomicon-admin-archive'
        ];
        $data = array_merge($this->getSorted(Mea::type(Type::AFFICHE)->published()->get()), $actions);

        $data['type'] = Type::AFFICHE;
        $data['soustitre'] = 'À l\'affiche';

        return view('Mea::admin.listing', $this->getViewData($data));
    }

    public function video()
    {
        $actions = $this->getDefaultActions();

        $type = Type::VIDEO;
        $data = array_merge($this->getSorted(Mea::type($type)->get()), $actions);
        $data['type'] = $type;
        $data['soustitre'] = $this->getSousTitreFromType($type);
        //$data['mea'] = ;

        if (config('mea.module_video') == true) {
            return view('Mea::admin.listing', $this->getViewData($data));
        } else { // Module video inactif => Redirection vers le principal
            $la_route = $this->getRouteFromType(Type::PRINCIPAL);
            return redirect()->route($la_route);
        }
    }

    public function bandeaux()
    {
        $actions = $this->getDefaultActions();

        // On ajoute des actions au sous-menu
        $actions['sub_actions'][1] = [
            'name' => 'Dépublier', 'href' => 'admin_mea_unpublish', 'picto' => 'bloomicon-admin-arrow-down-left'
        ];
        $actions['sub_actions'][2] = [
            'name' => 'Archiver', 'href' => 'admin_mea_archive', 'picto' => 'bloomicon-admin-archive'
        ];
        $data = array_merge($this->getSorted(Mea::type(Type::BANDEAUX)->published()->get()), $actions);

        $data['type'] = Type::BANDEAUX;
        $data['soustitre'] = 'Bandeaux';

        if (config('mea.module_bandeaux') == true) {
            return view('Mea::admin.listing', $this->getViewData($data));
        } else { // Module bandeaux inactif => Redirection vers le principal
            $la_route = $this->getRouteFromType(Type::BANDEAUX);
            return redirect()->route($la_route);
        }

        return view('Mea::admin.listing', $this->getViewData($data));
    }

    public function programmed()
    {
        $actions = $this->getDefaultActions();

        // On ajoute des actions au sous-menu
        $actions['sub_actions'][1] = [
            'name' => 'Publier', 'href' => 'admin_mea_publish', 'picto' => 'bloomicon-admin-arrow-up-right'
        ];
        $actions['sub_actions'][2] = [
            'name' => 'Archiver', 'href' => 'admin_mea_archive', 'picto' => 'bloomicon-admin-archive'
        ];

        $type = request('type', Type::PRINCIPAL);
        $data = array_merge($this->getSorted(Mea::type($type)->programmed()->get()), $actions);

        $data['soustitre'] = $this->getSousTitreFromType($type);

        return view('Mea::admin.listing', $this->getViewData($data));
    }

    public function encours()
    {
        $actions = $this->getDefaultActions();

        // On ajoute des actions au sous-menu
        $actions['sub_actions'][1] = [
            'name' => 'Publier', 'href' => 'admin_mea_publish', 'picto' => 'bloomicon-admin-arrow-up-right'
        ];
        $actions['sub_actions'][2] = [
            'name' => 'Archiver', 'href' => 'admin_mea_archive', 'picto' => 'bloomicon-admin-archive'
        ];

        $type = request('type', Type::PRINCIPAL);
        $data = array_merge($this->getSorted(Mea::type($type)->status(Statut::EN_COURS)->notProgrammed()->get()), $actions);

        $data['soustitre'] = $this->getSousTitreFromType($type);

        return view('Mea::admin.listing', $this->getViewData($data));
    }

    public function archived()
    {
        $type = request('type', Type::PRINCIPAL);

        $actions = new Actions();
        $actions
            ->addAction('Désarchiver', 'admin_mea_unpublish')
            ->addNewTabAction('Supprimer', 'admin_mea_delete');
        $actions = $actions->toArray();

        $actions['sub_actions'][0] = [
            'name' => 'Dupliquer', 'href' => 'admin_mea_duplicate', 'picto' => 'bloomicon-admin-copy'
        ];

        $data = array_merge($this->getSorted(Mea::type($type)->archived()->get()), $actions);

        $data['soustitre'] = $this->getSousTitreFromType($type);

        return view('Mea::admin.listing', $this->getViewData($data));
    }

    public function edit(Mea $mea)
    {
        $pages = Page::published()->get();
        $type = request('type', Type::PRINCIPAL);

        return view('Mea::admin.edit', compact('mea', 'pages', 'type'));
    }

    public function save(Mea $mea)
    {
        $user_id = Auth::user()->getAuthIdentifier();
        $type = request('type');

        if (request('statut_id')) {
            $statut = request('statut_id');
        } else {
            if ($mea !== null && $mea->count() > 0) {
                $statut = $mea->statut_id ?: Statut::EN_COURS;
            } else {
                $statut = Statut::EN_COURS;
            }
        }

        //Stockage des infos
        $mea->titre = request('titre');
        $mea->soustitre = request('soustitre');
        $mea->mentions = request('mentions');
        $mea->visuel_1 = request('visuel_1');
        $mea->visuel_2 = request('visuel_2');
        $mea->to_publish_at = request('date_pub');
        $mea->to_unpublish_at = request('date_depub');
        $mea->last_modified_by = $user_id;

        //Gestion du lien
        if (request('link_choice') && request('lien')) {

            if (request('page_id')) {
                $mea->page_id = request('page_id');
                $mea->href = null;
            }

            if (request('href')) {
                $parsed_link = parse_url(request('href'));
                $mea->href = empty($parsed_link['scheme']) ? 'http://' . request('href') : $mea->href = request('href');
                $mea->page_id = null;
            }
        } else {
            $mea->page_id = null;
            $mea->href = null;
        }


        //Gestion du statut en fonction de la date de publication
        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = Statut::EN_COURS;
        } elseif (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->lessThanOrEqualTo(now())) {
            $statut = Statut::PUBLIE;
        }


        if (!$mea->exists) {
            //Nouvelle MEA
            $mea->created_by = $user_id;

            // Type MEA
            $mea->type = $type;

            if (request('statut_id') == Statut::PUBLIE) {
                $statut = Statut::PUBLIE;
                $mea->to_publish_at = now();
            }
            $mea->statut_id = $statut;


            //Gestion du rank
            $mea->rank = $mea->getRank();

            $mea->save();
        } else {
            // Edition

            $mea->statut_id = $statut;

            $mea->save();
        }

        //Redirection
        if ($statut == Statut::EN_COURS) {
            if ($mea->to_publish_at) {
                return redirect()->route('admin_mea_programmes', $mea->type);
            }

            return redirect()->route('admin_mea_encours', $mea->type);
        }

        $la_route = $this->getRouteFromType($mea->type);

        return redirect()->route($la_route);
    }

    public function preview(Mea $mea)
    {
        $type = $mea->type;
        $meas = Mea::published()->orderBy('rank')->get();
        $last_mea = $meas->where('type', $type)->last();

        if ($meas->where('type', $type)->count() < Mea::MAX_RANK[$type]) {
            $mea->rank = $last_mea->rank + 1;
        } else {
            $keys = $meas->where('type', $type)->keys()->last();
            $meas = $meas->forget($keys);
            $mea->rank = 4;
        }
        $meas = $meas->push($mea);

        return response(view(config('mea.vue_preview'), compact('meas')));
    }

    public function publish(Mea $mea)
    {
        $mea->publish();

        return redirect()->back();
    }

    public function unpublish(Mea $mea)
    {
        $mea->unpublish();

        return redirect()->back();
    }

    public function archive(Mea $mea)
    {
        $mea->archive();

        return redirect()->back();
    }

    public function delete(Mea $mea)
    {
        $mea->softDelete();

        return redirect()->back();
    }

    public function duplicate(Mea $mea)
    {
        $mea2 = $mea->replicate();
        $mea2->titre .= ' - Copie';
        // L’entité dupliquée prend le statut ‘en cours’
        $mea2->statut_id = Statut::EN_COURS;
        $mea2->to_publish_at = null;
        $mea2->to_unpublish_at = null;
        $mea2->save();
        // Ouverture de l’onglet/workflow ‘en cours’
        return redirect()->route('admin_mea_encours', $mea->type);
    }

    public function reorder()
    {
        $ordre = request('ordre', []);

        $meas = Mea::published()->get();

        foreach ($meas as $mea) {
            /**
             * @var Mea $mea
             */
            $mea->rank = array_search($mea->id, $ordre, false);
            $mea->save();
        }
    }

    private function getTypeFromRoute($route): String
    {
        $type = Type::AFFICHE;

        if (request('type')) {
            $type = request('type');
        } else {
            if ($route === 'admin_mea_index') {
                $type = Type::PRINCIPAL;
            } else if ($route === 'admin_mea_video') {
                $type = Type::VIDEO;
            } else if ($route === 'admin_mea_bandeaux') {
                $type = Type::BANDEAUX;
            }
        }

        return $type;
    }

    private function getRouteFromType($type): String
    {
        $route = 'admin_mea_aff';

        if ($type == Type::PRINCIPAL) {
            $route = 'admin_mea_index';
        } else if ($type == Type::VIDEO) {
            $route = 'admin_mea_video';
        } else if ($type === Type::BANDEAUX) {
            $route = 'admin_mea_bandeaux';
        }

        return $route;
    }

    private function getSousTitreFromType($type): String
    {
        $soustitre = 'À l\'affiche';

        if ($type === Type::PRINCIPAL) {
            $soustitre = 'Visuel principal';
        } else if ($type === Type::VIDEO) {
            $soustitre = 'Paramétrer la vidéo de la page d\'accueil';
        } else if ($type === Type::BANDEAUX) {
            $soustitre = 'Bandeaux';
        }

        return $soustitre;
    }
}
