<?php
/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 25/06/20
 * Time: 12:13
 */

namespace Bloom\Cms\Leads\Model;


use Bloom\Cms\Leads\Extraction\ExtractionError;
use Bloom\Cms\Leads\Extraction\Extractor;
use Bloom\Cms\Leads\Notification\NotificationError;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;

abstract class Lead extends Model
{
    protected $extractors = null;
    protected $should_notify = false;


    public function onExtractorError(Lead $lead, ExtractionError $error, $name)
    {
        Log::critical('Error while exporting lead', ['lead' => $lead, 'error' => $error, 'extractor' => $name]);
    }

    public function onNotificationError(Lead $lead, NotificationError $error)
    {
        Log::error('Error while notifiying lead', ['lead' => $lead, 'error' => $error]);
    }

    protected static function booted()
    {
        static::created(function (Lead $lead) {
            if ($lead->should_notify) {
                $lead->notify();
            }
            if ($lead->extractors !== null) {
                $lead->extract();
            }
        });
        $instance = new static;
        if ($instance->extractors !== null) {
            foreach (static::buildEventList($instance) as $event => $handler) {
                static::listenIfExist($event, $handler);
            }
        }
        if ($instance->should_notify) {
            foreach ([
                         'lead.notification.sending' => static::class . '@onNotificationSending',
                         'lead.notification.sent'    => static::class . '@onNotificationSent',
                         'lead.notification.error'   => static::class . '@onNotificationError',
                     ] as $event => $handler) {
                static::listenIfExist($event, $handler);
            }
        }
    }

    private static function buildEventList(Lead $instance)
    {
        $list = [];
        $events = ['sending', 'sent', 'error'];
        foreach ($instance->extractors as $extractor) {
            foreach ($events as $event) {
                $list[ 'lead.extractor.' . $event . '.' . class_basename($extractor) ] = static::class . '@on' . class_basename($extractor) . ucfirst($event);
            }
        }
        foreach ($events as $event) {
            $list[ 'lead.extractor.' . $event ] = static::class . '@onExtractor' . ucfirst($event);
        }

        return $list;
    }

    private static function listenIfExist(string $event, string $handler)
    {
        list($class, $name) = explode('@', $handler);
        if (method_exists($class, $name)) {
            Event::listen($event, $handler);
        }
    }

    /**
     * @return void
     * @throws NotificationError
     */
    public abstract function sendNotififcation(): void;

    private function notify()
    {
        event('lead.notification.sending', $this);
        try {
            $this->sendNotififcation();
            event('lead.notification.sent', $this);
        } catch (NotificationError $e) {
            event('lead.notification.error', [$this, $e]);
        }
    }

    private function extract()
    {
        foreach ($this->extractors as $name) {
            $extractor = new $name;
            if ($extractor instanceof Extractor) {
                $extractor->send($this);
            } else {
                throw new \InvalidArgumentException('extractor should be a ' . Extractor::class);
            }
        }
    }
}
