<?php

namespace Bloom\Cms\Concession\Http\Controllers;

use App\Http\Controllers\Controller;
use Bloom\Cms\Concession\Helpers\ConcessionHelper;
use Bloom\Cms\Concession\Models\Concession;
use Bloom\Cms\Concession\Models\Membre;
use Bloom\Cms\Concession\Models\Service;
use Bloom\Cms\Concession\Repository\Eloquent\MembreRepository;
use Bloom\Cms\Concession\Repository\Eloquent\ServiceRepository;
use Bloom\CMS\Core\Contenus\Statut;
use Bloom\Cms\Sites\Helpers\SiteHelper;
use Bloom\Cms\Sites\Models\Site;
use Bloom\Cms\Sites\Repository\Eloquent\SiteRepository;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ConcessionController extends Controller
{

    public function index()
    {
        $user = Auth::user();
        if ($user->super_admin) {
            $concessions = Concession::all();
        } elseif ($user->groupe_site_id) {
            $sites = Site::query()->where('groupe_site_id', '=', $user->groupe_site_id)->get();
            $concessions = Concession::query()->whereIn('site_id', $sites->pluck('id')->toArray())->get();
        } else {
            $concessions = SiteHelper::filter(Concession::all(), false);
        }
        return view('Concession::index', compact('concessions'));
    }

    public function edit(Concession $concession, SiteRepository $siteRepository)
    {
        if ($concession === null) {
            $concession = new Concession();
        }
        $sites = null;
        if (config('sites.multi_sites')) {
            $sites = $siteRepository->getEnLigne();
        }

        return view('Concession::concession', compact('concession', 'sites'));
    }

    public function save(Concession $concession, Request $request)
    {
        $fields = [
            'nom_public' => 'required|max:45',
            'telephone' => 'required|max:12',
            'fax' => 'max:12',
            'coord_gps' => 'required',
            'visuel' => 'required',
            'horaires' => 'required',
            'description' => 'required',
            'mentions' => 'required',
        ];
        if (config('sites.multi_sites')) {
            $fields['site'] = 'required';
        }

        $validator = Validator::make($request->all(), $fields);
        if ($validator->fails()) {
            return back()
                ->withInput($request->all())
                ->withErrors($validator);
        } else {
            DB::beginTransaction();

            $concession->nom_public = $request->get('nom_public');
            $concession->telephone = $request->get('telephone');
            $concession->fax = $request->get('fax');
            $concession->coord_gps = $request->get('coord_gps');
            $concession->visuel = $request->get('visuel');
            $concession->horaires = $request->get('horaires');
            $concession->description = $request->get('description');
            $concession->mentions_legales = $request->get('mentions');
            $concession->knave_id = $request->get('knave_id');
            if (config('sites.multi_sites')) {
                $concession->site_id = $request->get('site');
            }

            if ($concession->save()) {
                DB::commit();
                session()->flash('message', 'Enregistrement ok');
            } else {
                DB::rollBack();
                session()->flash('error', "Problème d'enregistrement");
                Log::error('error', ['Une erreur est survenu lors de la modification des informations concession']);
            }
        }
        return redirect(route('admin_concession_index'));
    }

    public function services_index(Concession $concession, ServiceRepository $serviceRepository, Request $request)
    {
        if (is_null($concession)) {
            abort(404);
        }
        $type = $request->get('type') ?? 'published';
        switch ($type) {
            case 'encours':
                $services = $serviceRepository->getEncours($concession->id);
                break;
            case 'archived':
                $services = $serviceRepository->getArchived($concession->id);
                break;
            default:
                $services = $serviceRepository->getPublished($concession->id);
        }
        $nbPublished = $serviceRepository->getPublished($concession->id, true);
        $nbEncours = $serviceRepository->getEncours($concession->id, true);
        $nbArchived = $serviceRepository->getArchived($concession->id, true);

        return view(
            'Concession::liste_services',
            compact('services', 'concession', 'type', 'nbPublished', 'nbEncours', 'nbArchived')
        );
    }

    public function services_edit(Concession $concession, Service $service = null)
    {
        if (is_null($concession)) {
            abort(404);
        }
        $hasSav = ConcessionHelper::hasSav($concession);
        if ($service === null) {
            $service = new Service();
            $service->concession_id = $concession->id;
        }
        return view('Concession::service', compact('service', 'concession', 'hasSav'));
    }

    public function services_save(Request $request, Service $service)
    {
        $fields = [
            'nom' => 'required|max:45',
            'telephone' => 'required|max:12',
            'visuel' => 'required',
            'horaires' => 'required',
        ];

        $validator = Validator::make($request->all(), $fields);
        if ($validator->fails()) {
            return back()
                ->withInput(request()->all())
                ->withErrors($validator);
        } else {
            DB::beginTransaction();

            $service->nom = $request->get('nom');
            $service->telephone = $request->get('telephone');
            $service->fax = $request->get('fax');
            $service->sav = $request->get('sav');
            $service->visuel = $request->get('visuel');
            $service->horaires = $request->get('horaires');
            $service->description = $request->get('description');
            $service->concession_id = $request->get('concession_id');
            $service->concession_id = $request->get('concession_id');
            $user = auth()->user();
            if (!$service->created_by) {
                $service->created_by = $user->id;
            }
            $service->updated_by = $user->id;
            $service->statut_id = Statut::PUBLIE;

            if ($service->save()) {
                DB::commit();
                session()->flash('message', 'Enregistrement ok');
            } else {
                DB::rollBack();
                session()->flash('error', "Problème d'enregistrement");
                Log::error('error', ['Une erreur est survenu lors de l\'enregistrement du service']);
            }
        }
        return redirect(route('admin_concession_services_index', [$service->concession_id]));
    }

    public function services_publish(Service $service)
    {
        $service->statut_id = Statut::PUBLIE;
        $service->save();

        return redirect()->back();
    }

    public function services_unpublish(Service $service)
    {
        $service->statut_id = Statut::EN_COURS;
        $service->save();

        return redirect()->back();
    }

    public function services_archive(Service $service)
    {
        $service->statut_id = Statut::ARCHIVE;
        $service->save();

        return redirect()->back();
    }

    public function services_unarchive(Service $service)
    {
        $service->statut_id = Statut::PUBLIE;
        $service->save();

        return redirect()->back();
    }

    public function services_delete(Service $service)
    {
        $service->statut_id = Statut::SUPPRIMER;
        $service->save();

        return redirect()->back();
    }

    public function membres_index(Concession $concession, MembreRepository $membreRepository, Request $request)
    {
        if (is_null($concession)) {
            abort(404);
        }
        $type = $request->get('type') ?? 'published';
        switch ($type) {
            case 'encours':
                $membres = $membreRepository->getEncours($concession->id);
                break;
            case 'archived':
                $membres = $membreRepository->getArchived($concession->id);
                break;
            default:
                $membres = $membreRepository->getPublished($concession->id);
        }
        $nbPublished = $membreRepository->getPublished($concession->id, true);
        $nbEncours = $membreRepository->getEncours($concession->id, true);
        $nbArchived = $membreRepository->getArchived($concession->id, true);

        return view(
            'Concession::liste_membres',
            compact('membres', 'concession', 'type', 'nbPublished', 'nbEncours', 'nbArchived')
        );
    }

    public function membres_edit(Concession $concession, Membre $membre = null)
    {
        if (is_null($concession)) {
            abort(404);
        }
        if ($membre === null) {
            $membre = new Membre();
            $membre->concession_id = $concession->id;
        }
        $services = ConcessionHelper::availableServices($concession);

        return view('Concession::membre', compact('membre', 'concession', 'services'));
    }

    public function membres_save(Request $request, Membre $membre)
    {
        $fields = [
            'nom' => 'required|max:45',
            'fonction' => 'required|max:45',
            'service' => 'required',
        ];

        $validator = Validator::make($request->all(), $fields);
        if ($validator->fails()) {
            return back()
                ->withInput(request()->all())
                ->withErrors($validator);
        } else {
            DB::beginTransaction();

            $membre->nom = $request->get('nom');
            $membre->fonction = $request->get('fonction');
            $membre->photo = $request->get('photo');
            $membre->service_id = $request->get('service');
            $membre->telephone = $request->get('telephone');
            $membre->concession_id = $request->get('concession_id');
            $user = auth()->user();
            if (!$membre->created_by) {
                $membre->created_by = $user->id;
            }
            $membre->updated_by = $user->id;
            $membre->statut_id = Statut::PUBLIE;

            if ($membre->save()) {
                DB::commit();
                session()->flash('message', 'Enregistrement ok');
            } else {
                DB::rollBack();
                session()->flash('error', "Problème d'enregistrement");
                Log::error('error', ['Une erreur est survenu lors de la modification du membre']);
            }
        }
        return redirect(route('admin_concession_membres_index', [$membre->concession_id]));
    }

    public function membres_publish(Membre $membre)
    {
        $membre->statut_id = Statut::PUBLIE;
        $membre->save();

        return redirect()->back();
    }

    public function membres_unpublish(Membre $membre)
    {
        $membre->statut_id = Statut::EN_COURS;
        $membre->save();

        return redirect()->back();
    }

    public function membres_archive(Membre $membre)
    {
        $membre->statut_id = Statut::ARCHIVE;
        $membre->save();

        return redirect()->back();
    }

    public function membres_unarchive(Membre $membre)
    {
        $membre->statut_id = Statut::PUBLIE;
        $membre->save();

        return redirect()->back();
    }

    public function membres_delete(Membre $membre)
    {
        $membre->statut_id = Statut::SUPPRIMER;
        $membre->save();

        return redirect()->back();
    }
}
