<?php

/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 07/12/2020
 * Time: 11:59
 */

namespace Bloom\CMS\HPV\Cliniques\Http\Controllers;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Actions;
use Bloom\CMS\HPV\Cliniques\Model\Clinique;
use Bloom\CMS\HPV\Cliniques\Model\Telephone;
use Grimzy\LaravelMysqlSpatial\Types\Point;
use http\Env\Request;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;

class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiées',
                    'href'   => route('admin_clinique_index'),
                    'number' => Clinique::published()->count(),
                    'active' => request()->route()->getName() === 'admin_clinique_index',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_clinique_encours', ['orderBy' => 'updated_at', 'order' => 'desc']),
                    'number' => Clinique::query()->where('statut_id', '=', Statut::EN_COURS)->count(),
                    'active' => request()->route()->getName() === 'admin_clinique_encours',
                ],
                [
                    'title'  => 'Archivées',
                    'href'   => route('admin_clinique_archived', ['orderBy' => 'updated_at', 'order' => 'desc']),
                    'number' => Clinique::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_clinique_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(string $publie, string $action): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_clinique_edit')
            ->addAction($publie, $action, 'bloomicon-admin-arrow-up-right')
            ->addNewTabAction('Prévisualiser', 'admin_clinique_preview');
        $sub_actions = new Actions('sub_actions');
        $sub_actions
            ->addAction('Dupliquer', 'admin_clinique_duplicate', 'bloomicon-admin-copy')
            ->addAction('Archiver', 'admin_clinique_archive', 'bloomicon-admin-archive');

        return array_merge($actions->toArray(), $sub_actions->toArray());
    }

    public function getSorted(Collection $collection)
    {
        //@codeCoverageIgnoreStart
        // La fonction est correctement appelé par la collection mais n'est pas lu par le coverage report, du coup
        // on l'ignore pour pas qu'il nous disent que la fonction est pas couverte
        $sort = function (Clinique $clinique) {
            return Arr::get(Arr::dot($clinique->toArray()), request('orderBy', 'id'));
        };
        //@codeCoverageIgnoreEnd

        return [
            'cliniques' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order'    => [
                'item'  => request('orderBy', 'id'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions('Dépublier', 'admin_clinique_unpublish');

        $data = array_merge($this->getSorted(Clinique::published()->get()), $actions);

        return view('Clinique::admin.index', $this->getViewData($data));
    }

    public function encours()
    {
        $actions = $this->getDefaultActions('Publier', 'admin_clinique_publish');

        $data = array_merge($this->getSorted(Clinique::where('statut_id', '=', Statut::EN_COURS)->get()), $actions);

        return view('Clinique::admin.index', $this->getViewData($data));
    }

    public function archived()
    {
        $actions = new Actions();
        $actions
            ->addNewTabAction('Prévisualiser', 'admin_clinique_preview')
            ->addAction('Désarchiver', 'admin_clinique_unpublish');

        $data = array_merge($this->getSorted(Clinique::archived()->get()), $actions->toArray());

        return view('Clinique::admin.index', $this->getViewData($data));
    }

    public function edit(Clinique $clinique)
    {
        $telephones = [];

        if (!empty($clinique->telephone)) {
            $telephones = $this->checkTelephonesIsEmpty($clinique->telephone);
        } else {
            $telephone['libelle'] = '';
            $telephone['value'] = '';
            $telephones[] = $telephone;
        }

        $clinique->telephone = $telephones;

        return view('Clinique::admin.edit', compact('clinique'));
    }

    public function checkTelephonesIsEmpty(array $telephones): array
    {
        $newTelephones = [];

        foreach ($telephones as $telephone) {
            if ($telephone['libelle'] != '') {
                $newTelephones[] = $telephone;
            }
        }
        return $newTelephones;
    }

    public function save(Clinique $clinique)
    {
        $clinique->nom = request('nom');
        $clinique->adresse = request('adresse');
        $clinique->cp = request('cp');
        $clinique->ville = request('ville');
        $clinique->telephone = $this->checkTelephonesIsEmpty(json_decode(request('values', '[]'), true));
        $clinique->fax = request('fax', '');
        $clinique->doctolib = request('doctolib');
        $clinique->vignette = request('vignette');
        $clinique->transport = request('transport');
        $clinique->description = request('description');
        $clinique->position = new Point(request('point_lat'), request('point_lng'));
        $clinique->statut_id = request('statut_id', Statut::EN_COURS);
        $clinique->save();

        if ((int)$clinique->statut_id === Statut::PUBLIE) {
            return redirect()->route('admin_clinique_index');
        }

        return redirect()->route('admin_clinique_encours');
    }

    //#region Actions

    public function preview(Clinique $clinique)
    {
        Log::debug("Loading clinique preview", compact('clinique'));
        $cliniques = Clinique::query()->whereIn('statut_id', [Statut::PUBLIE, Statut::EN_COURS])->get();

        return view('Clinique::front.index', compact('cliniques'));
    }

    public function duplicate(Clinique $clinique)
    {
        $dub = $clinique->replicate(['created_at', 'updated_at', 'created_by', 'last_modified_by']);
        $dub->nom = $clinique->nom . ' - Copie';
        $dub->save();

        return redirect()->back();
    }

    public function publish(Clinique $clinique)
    {
        $clinique->statut_id = Statut::PUBLIE;
        $clinique->to_publish_at = now();
        $clinique->save();

        return redirect()->back();
    }

    public function unpublish(Clinique $clinique)
    {
        $clinique->statut_id = Statut::EN_COURS;
        $clinique->to_unpublish_at = now();
        $clinique->save();

        return redirect()->back();
    }

    public function archive(Clinique $clinique)
    {
        $clinique->statut_id = Statut::ARCHIVE;
        $clinique->to_unpublish_at = now();
        $clinique->save();

        return redirect()->back();
    }

    public function delete(Clinique $clinique)
    {
        $clinique->statut_id = Statut::SUPPRIMER;
        $clinique->save();

        return redirect()->back();
    }
    //#endregion
}
