<?php

/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 07/12/2020
 * Time: 12:10
 */

namespace Tests\Feature;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Test;
use Bloom\CMS\HPV\Cliniques\CliniquesServiceProvider;
use Bloom\CMS\HPV\Cliniques\Model\Clinique;
use Grimzy\LaravelMysqlSpatial\Types\Point;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Auth\User;

class AdminCliniqueTest extends Test
{
    protected $provider = CliniquesServiceProvider::class;

    public function test_post_save_should_create_a_clinique()
    {
        /**
         * @var Clinique $clinique
         */
        $clinique = factory(Clinique::class)->make();

        $form = $clinique->toArray();
        /**
         * @var Point $point
         */
        $point = $form['position'];
        unset($form['position']);
        $form['point_lat'] = $point->getLat();
        $form['point_lng'] = $point->getLng();
        $form['values'] = json_encode($form['telephone']);
        unset($form['telephone']);

        $response = $this->actingAs(User::first())->post(route('admin_clinique_save'), $form);

        $expected = $clinique->makeHidden(['position', 'created_by', 'last_updated_by'])->toArray();
        $expected['telephone'] = json_encode($expected['telephone']);
        $response->assertRedirect(route('admin_clinique_index'));
        $this->assertDatabaseHas(
            'cliniques',
            $expected
        );
    }

    public function test_post_save_should_edit_a_clinique()
    {
        /**
         * @var Clinique $clinique
         */
        $clinique = factory(Clinique::class)->create();

        $form = $clinique->toArray();
        /**
         * @var Point $point
         */
        $point = $form['position'];
        unset($form['position']);
        $form['point_lat'] = $point->getLat();
        $form['point_lng'] = $point->getLng();

        $form['nom'] = "Clinique test";
        $form['values'] = json_encode($form['telephone']);
        unset($form['telephone']);

        $expected = $clinique->makeHidden(['position', 'created_by', 'last_updated_by', 'updated_at'])->toArray();
        $expected['nom'] = "Clinique test";
        $expected['telephone'] = json_encode($expected['telephone']);

        $response = $this->actingAs(User::first())->post(route('admin_clinique_save', $clinique), $form);

        $response->assertRedirect(route('admin_clinique_index'));
        $this->assertDatabaseHas(
            'cliniques',
            $expected
        );
    }

    public function statutProvider()
    {
        return [
            [
                Statut::PUBLIE,
                'admin_clinique_index'
            ],
            [
                Statut::EN_COURS,
                'admin_clinique_encours'
            ],
            [
                Statut::ARCHIVE,
                'admin_clinique_archived'
            ],
        ];
    }

    /**
     * @param int    $statut
     * @param string $route
     *
     * @dataProvider statutProvider
     */
    public function test_index_should_show_current_statut_only(int $statut, string $route)
    {
        $expected = factory(Clinique::class, rand(1, 10))->create(['statut_id' => $statut]);
        // On ajoute des cliniques supprimer
        factory(Clinique::class, rand(1, 10))->create(['statut_id' => Statut::SUPPRIMER]);
        // On ajoute des cliniques pas du même statut que celui attendu
        factory(Clinique::class, rand(1, 10))
            ->create(['statut_id' => $statut === Statut::ARCHIVE ? Statut::PUBLIE : Statut::ARCHIVE]);

        $response = $this->actingAs(User::first())->get(route($route));
        $response->assertOk();
        $response->assertViewIs('Clinique::admin.index');
        /**
         * @var Clinique[]|Collection $cliniques
         */
        $cliniques = $response->viewData('cliniques');

        static::assertEquals($expected->count(), $cliniques->count());
        foreach ($cliniques as $clinique) {
            static::assertEquals($statut, (int)$clinique->statut_id);
        }
    }
}
