<?php

/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Articles\Http\Controllers;

use Bloom\CMS\Core\Helpers\Actions;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Articles\Model\Article;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;

/**
 * Class AdminController
 *
 * @package Bloom\CMS\Modules\Articles\Http\Controllers
 */
class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiés',
                    'href'   => route('admin_article_index'),
                    'number' => Article::published()->count(),
                    'active' => request()->route()->getName() === 'admin_article_index',
                ],
                [
                    'title'  => 'Programmés',
                    'href'   => route('admin_article_programmes'),
                    'number' => Article::programmed()->count(),
                    'active' => request()->route()->getName() === 'admin_article_programmes',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_article_encours'),
                    'number' => Article::status(2)->notProgrammed()->count(),
                    'active' => request()->route()->getName() === 'admin_article_encours',
                ],
                [
                    'title'  => 'Archivés',
                    'href'   => route('admin_article_archived'),
                    'number' => Article::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_article_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_article_edit')
            ->addNewTabAction('Prévisualiser', 'admin_article_preview')
            ->addAction('Nouvelle version', '');
        $sub_actions = new Actions('sub_actions');
        $sub_actions
            ->addAction('Dupliquer', 'admin_article_duplicate', 'bloomicon-admin-copy')
            ->addAction('Publier', 'admin_article_publish', 'bloomicon-admin-arrow-up-right')
            ->addAction('Archiver', 'admin_article_archive', 'bloomicon-admin-archive');

        return array_merge($actions->toArray(), isset($sub_actions) ? $sub_actions->toArray() : []);
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');
        //@codeCoverageIgnoreStart
        // La fonction est correctement appelé par la collection mais n'est pas lu par le coverage report, du coup
        // on l'ignore pour pas qu'il nous disent que la fonction est pas couverte
        $sort = function (Article $article) {
            return Arr::get(Arr::dot($article->toArray()), request('orderBy', 'id'));
        };

        //@codeCoverageIgnoreEnd

        return [
            'articles' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order'    => [
                'item'  => request('orderBy', 'id'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions();

        // On remplace la 2ieme action du menu
        $actions['sub_actions'][1] = [
            'name' => 'Dépublier', 'href' => 'admin_article_unpublish'
        ];

        $data = array_merge($this->getSorted(Article::published()->get()), $actions);

        return view('Article::admin.listing', $this->getViewData($data));
    }

    public function programmed()
    {
        $data = array_merge($this->getSorted(Article::programmed()->get()), $this->getDefaultActions());

        return view('Article::admin.listing', $this->getViewData($data));
    }

    public function encours()
    {
        $data = array_merge($this->getSorted(Article::status(2)->notProgrammed()->get()), $this->getDefaultActions());

        return view('Article::admin.listing', $this->getViewData($data));
    }

    public function archived()
    {
        $data = array_merge(
            $this->getSorted(Article::archived()->get()),
            (new Actions())
                ->addAction('Désarchiver', 'admin_article_unpublish')
                ->addAction('Supprimer', 'admin_article_delete')->toArray()
        );

        return view('Article::admin.listing', $this->getViewData($data));
    }

    /**
     * @param \Traversable|Dossier[] $dossiers
     * @param int $level
     * @return \Illuminate\Support\Collection
     */
    public function sortDossiers(\Traversable $dossiers, int $level): \Illuminate\Support\Collection
    {
        $tab = collect();

        foreach ($dossiers as $dossier) {
            $dossier->level = $level;
            $tab[] = $dossier;
            if ($dossier->childs) {
                $tab = $tab->concat($this->sortDossiers($dossier->childs, $level + 1));
            }
        }
        return $tab;
    }

    public function edit(Article $article)
    {
        $tableDossiers = Dossier::query()->whereNull('parent_id')->get();

        $page = $article->page;
        if (request('dossier_id')) {
            if ($page === null) {
                $page = new Page();
            }
            $page->dossier_id = request('dossier_id');
        }

        $dossiers = $this->sortDossiers($tableDossiers, 0);

        return view('Article::admin.edit', compact('article', 'dossiers', 'page'));
    }

    /**
     * Permet de fill les parametre générique d'une page avec le formulaire
     *
     * @param Page $page
     */
    protected function fillPage(Page $page): void
    {
        $page->titre = request('titre');
        $page->slug = request('slug');
        $page->intro = request('intro');
        $page->dossier_id = request('dossier_id');
        $page->to_publish_at = request('date_pub');
        $page->to_unpublish_at = request('date_depub');
        $page->image_opengraph = request('vignette');
        $page->mentions = request('mentions');
    }

    /**
     * Crée l'article et sa page en valorisant correctement le statut, celui-ci pouvant être surchargé en fonction
     * des valeurs du formulaire
     *
     * @param Article $article
     * @param int     $statut
     *
     * @return int
     */
    public function createArticle(Article $article, int $statut): int
    {
        // On crée l'article
        $article->contenu = request('contenu');
        $article->save();

        // On crée la page
        $page = new Page();
        $page->meta_titre = '';
        $page->meta_description = '';
        $page->created_by = Auth::user()->getAuthIdentifier();
        $page->last_modified_by = Auth::user()->getAuthIdentifier();
        $page->multiplicity = Page::SINGLE;
        $this->fillPage($page);

        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = 2;
        }
        if (request('statut_id') == 1) {
            $statut = 1;
            $page->to_publish_at = now();
        }

        $page->statut_id = $statut;
        // On attache la page et l'article
        $page->contenu()->associate($article);
        $page->save();

        return $statut;
    }

    /**
     * Modifie l'article et sa page en valorisant correctement le statut, celui-ci pouvant être surchargé en fonction
     * des valeurs du formulaires
     *
     * @param Article $article
     * @param int     $statut
     *
     * @return int
     */
    public function editArticle(Article $article, int $statut): int
    {
        /**
         * @var Page $page
         */
        $page = $article->page;

        $article->contenu = request('contenu');
        if ($article->isDirty()) {
            // On mets à jour la page
            $page->last_modified_by = Auth::user()->getAuthIdentifier();
            $article->save();
        }

        $this->fillPage($page);

        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = 2;
        }
        if (request('statut_id') == 1) {
            $statut = 1;
            $page->to_publish_at = now();
        }
        $page->statut_id = $statut;

        if ($page->isDirty()) {
            $page->last_modified_by = Auth::user()->getAuthIdentifier();
            $page->save();
            $article->touch();
        }

        return $statut;
    }

    public function save(Article $article)
    {
        $statut = 2;

        if ($article->page !== null) {
            $statut = $article->page->statut_id ?: request('statut_id', 2);
        }
        $statut = $article->exists ? $this->editArticle($article, $statut) : $this->createArticle($article, $statut);

        if ($statut == 2) {
            // on reset la relation page pour la vérification du publish at
            $article->unsetRelation('page');
            if ($article->page->to_publish_at) {
                return redirect()->route('admin_article_programmes');
            }

            return redirect()->route('admin_article_encours');
        }

        return redirect()->route('admin_article_index');
    }

    public function preview(Article $article)
    {
        return $article->getHandler()->single($article);
    }

    public function archive(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->archive();

        return redirect()->back();
    }

    public function publish(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->publish();

        return redirect()->back();
    }

    public function unpublish(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->unpublish();

        return redirect()->back();
    }

    public function delete(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->softDelete();

        return redirect()->back();
    }

    public function duplicate(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page2 = $page->replicate();
        $page2->titre .= ' - Copie';
        $page2->slug .= '-copie';
        $page2->pathname .= '-copie';
        $page2->save();

        $article2 = $article->replicate();
        $article2->save();
        $page2->contenu()->associate($article2);
        $page2->save();

        return redirect()->back();
    }
}
