<?php
/**
 * Created by : loic
 * Created at : 22/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Tests\Feature;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Workflow;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Articles\CmsArticlesServiceProvider;
use Bloom\CMS\Modules\Articles\Model\Article;
use Faker\Factory;
use Illuminate\Foundation\Auth\User;

/**
 * Class ArticleTest
 * @package Tests\Feature
 */
class ArticleTest extends \Bloom\CMS\Core\Test
{
    protected $provider = CmsArticlesServiceProvider::class;

    public function test_index_articles_should_show_articles()
    {
        $response = $this->actingAs(User::first())->get(route('admin_article_index'));

        $response->isOk();
        $response->assertViewIs('Core::admin.listing');
        $response->assertViewHas('contents');
    }

    public function test_edit_article_should_show_form()
    {
        $response = $this->actingAs(User::first())->get(route('admin_article_edit'));

        $response->isOk();
        $response->assertViewIs('Article::admin.edit');
    }

    public function test_edit_article_should_show_form_with_article()
    {
        /**
         * @var Article $article
         */
        $article = factory(Article::class)->create();
        $article->page = factory(Page::class)->create();

        $response = $this->actingAs(User::first())->get(route('admin_article_edit', $article));

        $response->isOk();
        $response->assertViewIs('Article::admin.edit');

        /**
         * @var Article $actual
         */
        $actual = $response->viewData('article');

        $this->assertInstanceOf(Article::class, $actual);
        $this->assertTrue($actual->exists);
    }

    public function form_provider()
    {
        $this->setUp();
        $faker = Factory::create();
        $array = [];
        for ($i = 0; $i < 10; $i++) {
            $array[] = [
                [ // Form
                    'titre'      => $faker->text(50),
                    'slug'       => $faker->slug,
                    'vignette'   => 'vignette.jpg',
                    // Les html ont un \n qui est trim par Laravel, donc on le supprime directement dans la source
                    'contenu'    => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'intro'      => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'dossier_id' => null,
                    'statut_id'  => 1
                ]
            ];
        }

        return $array;
    }

    /**
     * @param array $form
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_save_article(array $form)
    {
        $response = $this->actingAs(User::first())->post(route('admin_article_save'), $form);

        $response->isOk();
        $article = Article::first();

        $this->assertDatabaseHas('articles', [
            'contenu' => $form['contenu']
        ]);
        $this->assertDatabaseHas('pages', [
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'intro'           => $form['intro'],
            'image_opengraph' => $form['vignette'],
            'contenu_type'    => 'articles',
            'contenu_id'      => $article->id
        ]);
    }


    /**
     * @param array $form
     *
     * @depends      test_post_edit_should_save_article
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_edit_article(array $form)
    {
        // Sa permet de crée l'article / page necessaire sans ce prendre la tête a tout réecrire
        $this->test_post_edit_should_save_article($form);

        /**
         * @var Article $article
         * @var Page    $page
         */
        $article = Article::first();
        $page = $article->page;

        // On modifie des trucs
        $form['titre'] = 'test des truc';
        $form['slug'] = 'test-des-truc';
        // On fait comme si l'image n'était pas touché
        $form['image'] = $page->image_opengraph;


        $response = $this->actingAs(User::first())->post(route('admin_article_save', $article), $form);
        $response->isOk();

        $this->assertDatabaseHas('articles', [
            'contenu' => $form['contenu']
        ]);

        $this->assertDatabaseHas('pages', [
            'id'              => $page->id,
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'image_opengraph' => $form['image']
        ]);
    }

    public function test_preview_should_a_front_page()
    {
        $article = \factory(Article::class)->create();
        $page = \factory(Page::class)->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_preview', $article));
        $response->assertOk();
        $response->assertViewIs('Article::front.index');
        static::assertInstanceOf(Page::class, $response->viewData('page'));
        static::assertEquals($page->id, $response->viewData('page')->id);
        static::assertInstanceOf(Article::class, $response->viewData('article'));
        static::assertEquals($article->id, $response->viewData('article')->id);
    }

    public function test_preview_should_a_front_page_even_if_not_published()
    {
        $article = \factory(Article::class)->create();
        $page = \factory(Page::class)->state('unpublished')->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_preview', $article));
        $response->assertOk();
        $response->assertViewIs('Article::front.index');
        static::assertInstanceOf(Page::class, $response->viewData('page'));
        static::assertEquals($page->id, $response->viewData('page')->id);
        static::assertInstanceOf(Article::class, $response->viewData('article'));
        static::assertEquals($article->id, $response->viewData('article')->id);
    }

    public function test_preview_should_401_if_not_connected()
    {
        $article = \factory(Article::class)->create();
        $page = \factory(Page::class)->state('unpublished')->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);
        // On force le content type attendu sur du json pour pas être redirigé vers le login, et avoir le bon statut
        $response = $this->get(route('admin_article_preview', $article), ['Accept' => 'application/json']);
        $response->assertUnauthorized();
    }

    public function test_programmed_should_filtered_corectly()
    {
        $expected_workflow = Workflow::query()->where('code' ,'=', 'ARTICLE_PROGRAMMED')->first();
        static::assertInstanceOf(Workflow::class, $expected_workflow);

        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)->state('unpublished')->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_index', $expected_workflow));
        $response->assertOk();
        $response->assertViewIs('Core::admin.listing');
        $workflow = $response->viewData('workflow');
        static::assertEquals($expected_workflow->id, $workflow->id, 'Le workflow chargé n\'est pas le bon');
        $articles = $response->viewData('contents');
        static::assertEquals(0, $articles->count());

        $page->to_publish_at = now()->addWeek();
        $page->save();

        $response = $this->actingAs(User::first())->get(route('admin_article_index', $expected_workflow));
        $response->assertOk();
        $response->assertViewIs('Core::admin.listing');
        $workflow = $response->viewData('workflow');
        static::assertEquals($expected_workflow->id, $workflow->id, 'Le workflow chargé n\'est pas le bon');
        $articles = $response->viewData('contents');
        static::assertEquals(1, $articles->count());
    }

    public function test_encours_should_filtred_out_the_programmed_one()
    {
        $expected_workflow = Workflow::query()->where('code' ,'=', 'ARTICLE_EN_COURS')->first();
        static::assertInstanceOf(Workflow::class, $expected_workflow);

        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)->state('unpublished')->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_index',$expected_workflow));
        $response->assertOk();
        $response->assertViewIs('Core::admin.listing');
        $workflow = $response->viewData('workflow');
        static::assertEquals($expected_workflow->id, $workflow->id, 'Le workflow chargé n\'est pas le bon');
        $articles = $response->viewData('contents');
        static::assertEquals(1, $articles->count());

        $page->to_publish_at = now()->addWeek();
        $page->save();

        $response = $this->actingAs(User::first())->get(route('admin_article_index', $expected_workflow));
        $response->assertOk();
        $response->assertViewIs('Core::admin.listing');
        $workflow = $response->viewData('workflow');
        static::assertEquals($expected_workflow->id, $workflow->id, 'Le workflow chargé n\'est pas le bon');
        $articles = $response->viewData('contents');
        static::assertEquals(0, $articles->count());
    }

    public function test_archived_should_only_show_archived_one()
    {
        $expected_workflow = Workflow::query()->where('code' ,'=', 'ARTICLE_ARCHIVED')->first();
        static::assertInstanceOf(Workflow::class, $expected_workflow);
        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        \factory(Page::class)
            ->create(['contenu_type' => 'articles', 'contenu_id' => $article->id, 'statut_id' => 0]);

        $response = $this->actingAs(User::first())->get(route('admin_article_index', $expected_workflow));
        $response->assertOk();
        $response->assertViewIs('Core::admin.listing');
        $workflow = $response->viewData('workflow');
        static::assertEquals($expected_workflow->id, $workflow->id, 'Le workflow chargé n\'est pas le bon');
        $articles = $response->viewData('contents');
        static::assertEquals(1, $articles->count());

        // On ajoute un article publié pour voir
        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        \factory(Page::class)
            ->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_index', $expected_workflow));
        $response->assertOk();
        $response->assertViewIs('Core::admin.listing');
        $workflow = $response->viewData('workflow');
        static::assertEquals($expected_workflow->id, $workflow->id, 'Le workflow chargé n\'est pas le bon');
        $articles = $response->viewData('contents');
        static::assertEquals(1, $articles->count());
    }

    public function test_shouldnt_show_deleted_article()
    {
        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        \factory(Page::class)
            ->create(['contenu_type' => 'articles', 'contenu_id' => $article->id, 'statut_id' => Statut::SUPPRIMER]);

        foreach (Workflow::query()->get() as $workflow) {
            $response = $this->actingAs(User::first())->get(route('admin_article_index', $workflow));
            $response->assertOk();
            $response->assertViewIs('Core::admin.listing');
            $articles = $response->viewData('contents');
            static::assertEquals(0, $articles->count());
        }
    }

    public function test_action_archive_should_archied_and_redirect()
    {
        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)
            ->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_archive', $article));
        $response->assertRedirect();

        $page->refresh();
        static::assertEquals(Statut::ARCHIVE, $page->statut_id);
    }

    public function test_action_publish_should_publish_and_redirect()
    {
        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)
            ->state('unpublished')
            ->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_publish', $article));
        $response->assertRedirect();

        $page->refresh();
        static::assertEquals(Statut::PUBLIE, $page->statut_id);
        static::assertNotEmpty($page->to_publish_at);
    }

    public function test_action_unpublish_should_unpublish_and_redirect()
    {
        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)
            ->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_unpublish', $article));
        $response->assertRedirect();

        $page->refresh();
        static::assertEquals(Statut::EN_COURS, $page->statut_id);
        static::assertEmpty($page->to_publish_at);
    }

    public function test_action_delete_should_softdelete_and_redirect()
    {
        $article = \factory(Article::class)->create();
        /**
         * @var Page $page
         */
        $page = \factory(Page::class)
            ->create(['contenu_type' => 'articles', 'contenu_id' => $article->id]);

        $response = $this->actingAs(User::first())->get(route('admin_article_delete', $article));
        $response->assertRedirect();

        $page->refresh();
        static::assertEquals(Statut::SUPPRIMER, $page->statut_id);
    }
}
