<?php
/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Articles\Http\Controllers;


use Bloom\CMS\Modules\Architecture\Helpers\Actions;
use Bloom\CMS\Modules\Architecture\Model\Dossier;
use Bloom\CMS\Modules\Architecture\Model\Page;
use Bloom\CMS\Modules\Articles\Model\Article;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;


class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiés',
                    'href'   => route('admin_article_index'),
                    'number' => Article::published()->count(),
                    'active' => request()->route()->getName() === 'admin_article_index',
                ],
                [
                    'title'  => 'Programmés',
                    'href'   => route('admin_article_programmes'),
                    'number' => Article::programmed()->count(),
                    'active' => request()->route()->getName() === 'admin_article_programmes',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_article_encours'),
                    'number' => Article::status(2)->notProgrammed()->count(),
                    'active' => request()->route()->getName() === 'admin_article_encours',
                ],
                [
                    'title'  => 'Archivés',
                    'href'   => route('admin_article_archived'),
                    'number' => Article::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_article_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(bool $with_sub = true): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_article_edit')
            ->addNewTabAction('Prévisualiser', 'admin_article_preview')
            ->addAction('Nouvelle version', '');
        if ($with_sub) {
            $sub_actions = new Actions('sub_actions');
            $sub_actions
                ->addAction('Dupliquer', 'admin_article_duplicate')
                ->addAction('Publier maintenant', 'admin_article_publish')
                ->addAction('Archiver', 'admin_article_archive');
        }

        return array_merge($actions->toArray(), isset($sub_actions) ? $sub_actions->toArray() : []);
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');
        $sort = function (Article $article) {
            return Arr::get(Arr::dot($article->toArray()), request('orderBy', 'id'));
        };

        return [
            'articles' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order'    => [
                'item'  => request('orderBy', 'id'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions();

        // On remplace la 2ieme action du menu
        $actions['sub_actions'][1] = [
            'name' => 'Dépublier', 'href' => 'admin_article_unpublish'
        ];

        $data = array_merge($this->getSorted(Article::published()->get()), $actions);

        return view('Articles::admin.listing', $this->getViewData($data));
    }

    public function programmed()
    {
        $data = array_merge($this->getSorted(Article::programmed()->get()), $this->getDefaultActions());

        return view('Articles::admin.listing', $this->getViewData($data));
    }

    public function encours()
    {
        $data = array_merge($this->getSorted(Article::status(2)->notProgrammed()->get()), $this->getDefaultActions());

        return view('Articles::admin.listing', $this->getViewData($data));
    }

    public function archived()
    {
        $data = array_merge(
            $this->getSorted(Article::archived()->get()),
            (new Actions())
                ->addAction('Désarchiver', 'admin_article_unpublish')
                ->addAction('Supprimer', 'admin_article_delete')->toArray()
        );

        return view('Articles::admin.listing', $this->getViewData($data));
    }

    public function edit(Article $article)
    {
        $dossiers = Dossier::all();
        $page = null;
        if ($article->page->count()) {
            $page = $article->page;
        }

        return view('Articles::admin.edit', compact('article', 'dossiers', 'page'));
    }

    public function save(Article $article)
    {
        $user_id = Auth::user()->getAuthIdentifier();

        if (request('statut_id')) {
            $statut = request('statut_id');
        } else {
            if ($article->page !== null && $article->page->count() > 0) {
                $statut = $article->page->statut_id ?: 2;
            } else {
                $statut = 2;
            }
        }

        if (!$article->exists) {
            // Nouveau

            // On crée l'article
            $article->contenu = request('contenu');
            $article->save();

            // On crée la page
            $page = new Page();
            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->meta_titre = '';
            $page->meta_description = '';
            $page->created_by = $user_id;
            $page->last_modified_by = $user_id;
            $page->intro = request('intro');
            $page->dossier_id = request('dossier_id');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->multiplicity = Page::SINGLE;
            $page->image_opengraph = request('vignette');

            if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
                $statut = 2;
            }
            if (request('statut_id') == 1) {
                $statut = 1;
                $page->to_publish_at = now();
            }

            $page->statut_id = $statut;
            $page->save();

            // On attache la page et l'article
            $article->page()->attach($page->getKey());
        } else {
            // Edition

            /**
             * @var Page $page
             */
            $page = $article->page;

            $article->contenu = request('contenu');
            if ($article->isDirty()) {
                // On mets à jour la page
                $page->last_modified_by = $user_id;
                $article->save();
            }

            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->intro = request('intro');
            $page->dossier_id = request('dossier_id');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->image_opengraph = request('vignette');

            if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
                $statut = 2;
            }
            if (request('statut_id') == 1) {
                $statut = 1;
                $page->to_publish_at = now();
            }
            $page->statut_id = $statut;

            if ($page->isDirty()) {
                $page->last_modified_by = $user_id;
                $page->save();
                $article->touch();
            }
        }

        if ($statut == 2) {
            if ($page->to_publish_at) {
                return redirect()->route('admin_article_programmes');
            }

            return redirect()->route('admin_article_encours');
        }

        return redirect()->route('admin_article_index');
    }

    public function preview(Article $article)
    {
        return $article->getHandler()->single($article);
    }

    public function archive(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->archive();

        return redirect()->back();
    }

    public function publish(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->publish();

        return redirect()->back();
    }

    public function unpublish(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->unpublish();

        return redirect()->back();
    }

    public function delete(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->softDelete();

        return redirect()->back();
    }

    public function duplicate(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page2 = $page->replicate();
        $page2->titre .= ' - Copie';
        $page2->save();

        $article2 = $article->replicate();
        $article2->save();
        $article2->page()->attach($page2);

        return redirect()->back();
    }

}