<?php

/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Articles\Http\Controllers;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Workflow;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Articles\Model\Article;
use Bloom\Cms\Sites\Repository\Eloquent\SiteRepository;
use Carbon\Carbon;
use Illuminate\Routing\Controller;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Traversable;

/**
 * Class AdminController
 *
 * @package Bloom\CMS\Modules\Articles\Http\Controllers
 */
class AdminController extends Controller
{

    /**
     * @param Traversable|Dossier[] $dossiers
     * @param int $level
     * @return Collection
     */
    public function sortDossiers(Traversable $dossiers, int $level): Collection
    {
        $tab = collect();

        foreach ($dossiers as $dossier) {
            $dossier->level = $level;
            $tab[] = $dossier;
            if ($dossier->childs) {
                $tab = $tab->concat($this->sortDossiers($dossier->childs, $level + 1));
            }
        }
        return $tab;
    }

    public function edit(Article $article, SiteRepository $siteRepository)
    {
        $tableDossiers = Dossier::query()->whereNull('parent_id')->get();

        $page = $article->page;
        if (request('dossier_id')) {
            if ($page === null) {
                $page = new Page();
            }
            $page->dossier_id = request('dossier_id');
        }
        $sites = $site_pages = null;
        if (config('sites.multi_sites')) {
            $sites = $siteRepository->getEnLigne();
            if ($page) {
                $site_pages = $page->sites()->get();
            }
        }

        $dossiers = $this->sortDossiers($tableDossiers, 0);
        $path_restrict = config('articles.path_restrict');
        if ($path_restrict) {
            $user = Auth::user();
            if ($user->getAccessLevel() === 2) {
                $path = config('articles.path_restrict_honda');
                $dossiers = Dossier::where('slug', '=', $path)->get();

            } elseif ($user->getAccessLevel() > 2) {
                $path = config('articles.path_restrict_concess');
                $dossiers = Dossier::where('slug', '=', $path)->get();
            }
        }

        return view('Article::admin.edit', compact('article', 'dossiers', 'page', 'sites', 'site_pages', 'path_restrict'));
    }

    /**
     * Permet de fill les parametre générique d'une page avec le formulaire
     *
     * @param Page $page
     */
    protected function fillPage(Page $page): void
    {
        $page->titre = request('titre');
        $page->slug = request('slug');
        $page->intro = request('intro');
        $page->dossier_id = request('dossier_id');
        $page->to_publish_at = request('date_pub');
        $page->to_unpublish_at = request('date_depub');
        $page->image_opengraph = request('vignette');
        $page->mentions = request('mentions');
    }

    /**
     * Crée l'article et sa page en valorisant correctement le statut, celui-ci pouvant être surchargé en fonction
     * des valeurs du formulaire
     *
     * @param Article $article
     * @param int $statut
     * @return int
     */
    public function createArticle(Article $article, int $statut, SiteRepository $siteRepository): int
    {
        // On crée l'article
        $article->contenu = request('contenu');
        $article->save();

        // On crée la page
        $page = new Page();
        $page->meta_titre = '';
        $page->meta_description = '';
        $page->created_by = Auth::user()->getAuthIdentifier();
        $page->last_modified_by = Auth::user()->getAuthIdentifier();
        $page->multiplicity = Page::SINGLE;
        $this->fillPage($page);

        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = 2;
        }
        if (request('statut_id') == 1) {
            $statut = 1;
            $page->to_publish_at = now();
        }

        $page->statut_id = $statut;
        // On attache la page et l'article
        $page->contenu()->associate($article);
        $page->save();

        // gestion multisites
        $page->sites()->detach();
        $multisite = request('multisite');
        if (!empty($multisite)) {
            foreach ($multisite as $site_id) {
                $page->sites()->attach([
                    'sites_id' => $site_id,
                ]);
            }
        }
        $page->save();

        return $statut;
    }

    /**
     * Modifie l'article et sa page en valorisant correctement le statut, celui-ci pouvant être surchargé en fonction
     * des valeurs du formulaires
     *
     * @param Article $article
     * @param int $statut
     * @return int
     */
    public function editArticle(Article $article, int $statut, SiteRepository $siteRepository): int
    {
        /**
         * @var Page $page
         */
        $page = $article->page;

        $article->contenu = request('contenu');
        if ($article->isDirty()) {
            // On mets à jour la page
            $page->last_modified_by = Auth::user()->getAuthIdentifier();
            $article->save();
        }

        $this->fillPage($page);

        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = 2;
        }
        if (request('statut_id') == 1) {
            $statut = 1;
            $page->to_publish_at = now();
        }
        $page->statut_id = $statut;

        if ($page->isDirty()) {
            $page->last_modified_by = Auth::user()->getAuthIdentifier();
            $page->save();
            $article->touch();
        }

        // gestion multisites
        $page->sites()->detach();
        $multisite = request('multisite');
        if (!empty($multisite)) {
            foreach ($multisite as $site_id) {
                $page->sites()->attach([
                    'sites_id' => $site_id,
                ]);
            }
        }
        $page->save();

        return $statut;
    }

    public function save(Article $article, SiteRepository $siteRepository)
    {
        $statut = 2;

        if ($article->page !== null) {
            $statut = $article->page->statut_id ?: request('statut_id', 2);
        }
        $statut = $article->exists ? $this->editArticle($article, $statut, $siteRepository) : $this->createArticle($article, $statut, $siteRepository);

        if ($statut == 2) {
            // on reset la relation page pour la vérification du publish at
            $article->unsetRelation('page');
            if ($article->page->to_publish_at) {
                $workflow = Workflow::query()->where('code', '=', 'ARTICLE_PROGRAMMED')->first();
                return redirect()->route('admin_article_index', $workflow);
            }
            $workflow = Workflow::query()->where('code', '=', 'ARTICLE_EN_COURS')->first();

            return redirect()->route('admin_article_index', $workflow);
        }

        return redirect()->route('admin_article_index');
    }

    public function preview(Article $article)
    {
        return $article->getHandler()->single($article);
    }

    public function publish(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->publish();

        return redirect()->back();
    }

    public function unpublish(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->unpublish();

        return redirect()->back();
    }

    public function archive(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->archive();

        return redirect()->back();
    }

    public function duplicate(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page2 = $page->replicate();
        $page2->titre .= ' - Copie';
        $page2->slug .= '-copie';
        $page2->pathname .= '-copie';
        // L’entité dupliquée prend le statut ‘en cours’
        $page2->statut_id = Statut::EN_COURS;
        $page2->to_publish_at = null;
        $page2->to_unpublish_at = null;
        $page2->save();

        $article2 = $article->replicate();
        $article2->save();
        $page2->contenu()->associate($article2);
        $page2->save();
        // Ouverture de l’onglet/workflow ‘en cours’
        $workflow = Workflow::query()->where('code', '=', 'ARTICLE_EN_COURS')->first();

        return redirect()->route('admin_article_index', $workflow);
    }

    public function delete(Article $article)
    {
        /**
         * @var Page $page
         */
        $page = $article->page;
        $page->softDelete();

        return redirect()->back();
    }
}
