<?php
/**
 * Created by : loic
 * Created at : 22/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Tests\Feature;


use Bloom\CMS\Modules\Architecture\Model\Page;
use Bloom\CMS\Modules\Articles\Model\Article;
use Faker\Factory;
use Illuminate\Foundation\Auth\User;
use Illuminate\Http\Testing\File;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\Test;

class ArticleTest extends Test
{

    public function setUp(): void
    {
        parent::setUp();
        Storage::fake();
    }

    public function test_index_articles_should_show_articles()
    {
        $response = $this->actingAs(new User())->get(route('admin_article_index'));

        $response->isOk();
        $response->assertViewIs('Articles::admin.listing');
        $response->assertViewHas('articles');
    }

    public function test_edit_article_should_show_form()
    {
        $response = $this->actingAs(new User())->get(route('admin_article_edit'));

        $response->isOk();
        $response->assertViewIs('Articles::admin.edit');
    }

    public function test_edit_article_should_show_form_with_article()
    {
        /**
         * @var Article $article
         */
        $article = factory(Article::class)->create();
        $article->page = factory(Page::class)->create();

        $response = $this->actingAs(new User())->get(route('admin_article_edit', $article));

        $response->isOk();
        $response->assertViewIs('Articles::admin.edit');

        /**
         * @var Article $actual
         */
        $actual = $response->viewData('article');

        $this->assertInstanceOf(Article::class, $actual);
        $this->assertTrue($actual->exists);
    }

    public function form_provider()
    {
        $this->setUp();
        $faker = Factory::create();
        $array = [];
        for ($i = 0; $i < 10; $i++) {
            $array[] = [
                [ // Form
                    'titre'      => $faker->text(50),
                    'slug'       => $faker->slug,
                    'vignette'      => UploadedFile::fake()->image('vignette.jpg')->getClientOriginalName(),
                    // Les html ont un \n qui est trim par Laravel, donc on le supprime directement dans la source
                    'contenu'    => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'intro'      => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'dossier_id' => null,
                    'statut_id'  => 1
                ]
            ];
        }

        return $array;
    }

    /**
     * @param array $form
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_save_article(array $form)
    {

        $user = new User();
        $user->id = 1;
        $response = $this->actingAs($user)->post(route('admin_article_save'), $form);

        $response->isOk();

        $this->assertDatabaseHas('articles', [
            'contenu' => $form['contenu']
        ]);
        $this->assertDatabaseHas('pages', [
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'intro'           => $form['intro'],
            'image_opengraph' => $form['vignette']
        ]);

        $article = Article::first();

        $this->assertDatabaseHas('page_contenus', [
            'contenu_id' => $article->id,
            'page_id'    => $article->page->id
        ]);
    }


    /**
     * @param array $form
     *
     * @depends      test_post_edit_should_save_article
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_edit_article(array $form)
    {

        $user = new User();
        $user->id = 1;

        // Sa permet de crée l'article / page necessaire sans ce prendre la tête a tout réecrire
        $this->test_post_edit_should_save_article($form);

        /**
         * @var Article $article
         * @var Page    $page
         */
        $article = Article::first();
        $page = $article->page;

        // On modifie des trucs
        $form['titre'] = 'test des truc';
        $form['slug'] = 'test-des-truc';
        // On fait comme si l'image n'était pas touché
        $form['image'] = $page->image_opengraph;


        $response = $this->actingAs($user)->post(route('admin_article_save', $article), $form);
        $response->isOk();

        $this->assertDatabaseHas('articles', [
            'contenu' => $form['contenu']
        ]);

        $this->assertDatabaseHas('pages', [
            'id'              => $page->id,
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'image_opengraph' => $form['image']
        ]);

    }
}