<?php

namespace Tests\Unit;

use Bloom\CMS\Modules\Architecture\Database\Relation\BelongsToPage;
use Bloom\CMS\Modules\Architecture\Database\Relation\HasExtendedRelationships;
use Bloom\CMS\Modules\Architecture\Http\Controllers\ContenuHandler;
use Bloom\CMS\Modules\Architecture\Model\Page;
use Bloom\CMS\Modules\Architecture\Model\Statut;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Tests\Test;


/**
 * Class Contenu
 * @package Tests\Unit
 *
 * @property Page $page
 *
 *
 * @method static Builder|static published()
 * @method static Builder|static status(int $status)
 */
class Contenu extends Model implements \Bloom\CMS\Modules\Architecture\Model\Contenu
{
    use HasExtendedRelationships;

    protected $guarded = [];

    public function page(): BelongsToPage
    {
        return $this->belongsToPage();
    }

    public function scopePublished(Builder $query)
    {
        return $this->scopeStatus($query, 1);
    }

    public function scopeStatus(Builder $query, int $status)
    {
        return $query->whereHas('page', function (Builder $query) use ($status) {
            return $query->where('statut_id', '=', $status);
        });
    }


    public function getHandler(): ContenuHandler
    {
        return new class extends Controller implements ContenuHandler {
            public function single(\Bloom\CMS\Modules\Architecture\Model\Contenu $contenu): Response
            {
                return \response([
                    "status" => true
                ]);
            }

            public function multiple(Collection $contenus): Response
            {
                return \response();
            }

            public function ensemble()
            {
                return \response();
            }
        };
    }
}

class PageTest extends Test
{
    public function getContenu(array $attributes = []): Contenu
    {
        Relation::morphMap([
            'contenus' => Contenu::class,
        ]);

        Page::declareContenu('contenus');

        /**
         * @var Page $page
         */
        $page = factory(Page::class)->create($attributes);

        return $page->contenus()->create(
            ['html' => 'test']
        );
    }

    public function test_should_create_dynamic_polymorph_relation()
    {
        Relation::morphMap([
            'contenus' => Contenu::class,
        ]);

        Page::declareContenu('contenus');

        /**
         * @var Page $page
         */
        $page = factory(Page::class)->create();

        $a = $page->contenus()->create(
            ['html' => 'test']
        );

        $this->assertDatabaseHas('page_contenus', ['contenu_id' => $a->id, 'contenu_type' => 'contenus']);
    }

    public function test_should_auto_eager_loading_dynamiquye_contenu()
    {
        Relation::morphMap([
            'contenus' => Contenu::class,
        ]);

        Page::declareContenu('contenus');

        /**
         * @var Page $page
         */
        $page = factory(Page::class)->create();

        $a = $page->contenus()->create(
            ['html' => 'test']
        );

        $collections = $page->contenus;

        static::assertInstanceOf(Collection::class, $collections);

        $b = $collections->first();

        static::assertEquals($a->id, $b->id);
    }

    public function test_all_content_should_be_in_the_containers()
    {
        $contenu = $this->getContenu([
            'pathname' => '/test'
        ]);

        $page = $contenu->page;

        $containers = $page->containers;

        static::assertCount(1, $containers);

        static::assertEquals($contenu->id, $containers->first()->id);
    }

    public function test_eager_loading()
    {
        $this->getContenu([
            'pathname' => '/test'
        ]);
        $this->getContenu([
            'pathname' => '/test-2'
        ]);

        foreach (Contenu::with('page')->get() as $contenu) {
            static::assertNotNull($contenu->page);
        }
    }

    public function test_scope_existance()
    {
        $this->getContenu([
            'pathname'  => '/test',
            'statut_id' => Statut::PUBLIE
        ]);

        $a = Contenu::published()->get();

        static::assertNotEquals(0, $a->count());
    }

}
