<?php


namespace Bloom\CMS\Modules\Architecture\Model;


use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Str;

/**
 * Class Page
 * @package Bloom\CMS\Modules\Architecture\Model
 *
 * @property string               $titre
 * @property string               $meta_titre
 * @property string               $meta_description
 * @property string               $slug
 * @property string               $pathname
 * @property string               $intro
 * @property string               $multiplicity
 *
 * @property Dossier              $dossier
 * @property Statut               $statut
 * @property User                 $cree_par
 * @property User                 $deriere_modification_par
 * @property Collection|Contenu[] $containers
 *
 * @method static Builder|static published()
 * @method static Builder|static archived()
 */
class Page extends Model
{
    const SINGLE = '1';
    const MUTLI = '1-n';
    const ALL = '*';

    protected $guarded = [];


    public static function booted()
    {
        static::updating(function (Page $page) {
            if ($page->isDirty(['dossier_id'])) {
                if ($page->getOriginal('dossier_id') === null) {
                    $page->pathname = '/' . $page->dossier->slug . '/' . $page->slug;
                } else {
                    $original_dossier = Dossier::find($page->getOriginal('dossier_id'));
                    $page->pathname = str_replace($original_dossier->slug, $page->dossier->slug, $page->pathname);
                }
            }
        });
    }

    //#region Dynamique polymorphisme
    protected static $contenus = [];

    /**
     * Déclarer l'alias polymorphé d'un contenu pour qu'il soit acèssible dynamiquement
     *
     * @param string $contenu Alias de la class
     */
    public static function declareContenu(string $contenu): void
    {
        if (!in_array($contenu, array_keys(static::$contenus))) {
            // On check si sa implemente bien l'interface de contenu
            if (!is_subclass_of(Relation::getMorphedModel($contenu), Contenu::class)) {
                throw new \InvalidArgumentException("$contenu doit implémenté l'interface " . Contenu::class);
            }

            static::$contenus[ $contenu ] = function () use ($contenu) {
                return $this->morphedByMany(Relation::getMorphedModel($contenu), 'contenu', 'page_contenus', 'page_id', 'contenu_id', 'id', 'id');
            };
        }
    }

    public static function declareContenus(array $contenus): void
    {
        foreach ($contenus as $contenu) {
            static::declareContenu($contenu);
        }
    }

    public static function getDeclaredContenu(): array
    {
        return static::$contenus;
    }

    public function __call($method, $parameters)
    {
        if (in_array($method, array_keys(static::$contenus))) {
            $function = static::$contenus[ $method ];

            return $function->call($this);
        }

        // On doit surchargé l'appel en cas d'appel à une relation dynamique
        return parent::__call($method, $parameters);
    }

    public function getAttribute($key)
    {
        if (in_array($key, array_keys(static::$contenus))) {
            if ($this->relationLoaded($key)) {
                return $this->relations[ $key ];
            }

            $method = static::$contenus[ $key ];

            return tap($method->call($this)->getResults(),
                function ($results) use ($key) {
                    $this->setRelation($key, $results);
                }
            );
        }

        return parent::getAttribute($key);
    }

    //#endregion

    public function getContainersAttribute(): Collection
    {
        $containers = new Collection();

        foreach ($this->getConnection()->table('page_contenus')->where('page_id', '=', $this->{$this->primaryKey})->get() as $relation) {
            $class = Relation::getMorphedModel($relation->contenu_type);
            $containers->add($class::find($relation->contenu_id));
        }

        return $containers;
    }

    public function setMultiplicityAttribute(string $value)
    {
        $accepted = [static::SINGLE, static::MUTLI, static::ALL];
        if (!in_array($value, $accepted)) {
            throw new \InvalidArgumentException('multiplicity should be in ' . implode(',', $accepted));
        }

        $this->attributes['multiplicity'] = $value;
    }

    public function setSlugAttribute(string $slug)
    {
        $this->attributes['slug'] = Str::slug(str_replace('/', '', $slug));
    }

    public function dossier()
    {
        return $this->belongsTo(Dossier::class);
    }

    public function statut()
    {
        return $this->belongsTo(Statut::class);
    }

    public function cree_par()
    {
        return $this->belongsTo('user', 'created_by');
    }

    public function deriere_modification_par()
    {
        return $this->belongsTo('user', 'last_modified_by');
    }

    public function scopePublished(Builder $query)
    {
        return $query->where('statut_id', '=', 1);
    }

    public function scopeArchived(Builder $query)
    {
        return $query->where('statut_id', '=', 0);
    }
}