<?php


namespace Bloom\CMS\Modules\Architecture;


use Bloom\CMS\Modules\Architecture\Contenus\{Ensemble, Filtre};
use Bloom\CMS\Modules\Architecture\Http\Middleware\ModuleMiddleware;
use Bloom\CMS\Modules\Architecture\Model\{Module, Page};
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\{Route, Schema};
use Illuminate\Support\ServiceProvider;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;
use TwigBridge\Facade\Twig;

class CmsArchiServiceProvider extends ServiceProvider
{

    public function register(): void
    {
        $this->loadFactoriesFrom(dirname(__DIR__) . '/database/factories');
        $this->loadMigrationsFrom(dirname(__DIR__) . '/database/migrations');
        $this->loadViewsFrom(dirname(__DIR__) . '/resources/views', 'Archi');
        $this->registerRoutes();

        $this->publishes([
            dirname(__DIR__) . '/public' => public_path('/vendor/cms-module-archi')
        ], 'assets');
    }

    public function boot(): void
    {
        $this->addTwigExtension();
        $this->declareModule();
        $this->registerContenu();
        self::refreshRoutes();
    }

    public static function refreshRoutes(): void
    {
        if (Schema::hasTable('pages')) {
            Route::group(['namespace' => 'Bloom\CMS\Modules\Architecture\Http\Controllers', 'middleware' => ['web']], function () {
                foreach (Page::published()->get('pathname') as $page) {
                    Route::get($page->pathname, 'FrontController@handle');
                }
            });
        }
    }

    protected function registerRoutes(): void
    {
        $dirname = dirname(__DIR__) . '/routes';
        foreach (scandir($dirname) as $file) {
            if (!in_array($file, ['.', '..'])) {
                Route::group(['middleware' => ['web', ModuleMiddleware::class]], function () use ($dirname, $file) {
                    $this->loadRoutesFrom($dirname . '/' . $file);
                });
            }
        }
    }

    protected function declareModule(): void
    {
        if (Schema::hasTable('modules')) {
            $module = Module::nom('archi')->firstOrCreate([
                'nom'        => 'archi',
                'nom_public' => 'Architecture',
                'route'      => 'admin_archi_index',
            ]);
            if (!$module->exists) {
                $module->active = 1;
                $module->save();
            }
        }
    }

    protected function registerContenu(): void
    {

        $morphed = [
            'filtres'   => Filtre::class,
            'ensembles' => Ensemble::class,
        ];

        Relation::morphMap($morphed);

        Page::declareContenus($morphed);
    }

    protected function addTwigExtension(): void
    {
        Twig::addExtension(new class extends AbstractExtension {
            public function getFunctions()
            {
                return [
                    new TwigFunction('modules', function () {
                        return Module::actif()->get()->each(function (Module $module) {
                            $module->menu_actif = request()->module_actif == $module->id;
                        });
                    }),
                    new TwigFunction('route_name', function () {
                        return request()->route()->getName();
                    })
                ];
            }

        });
    }
}