<?php


namespace Bloom\CMS\Modules\Architecture\Model;


use Bloom\CMS\Modules\Architecture\Contenus\Contenu;
use Bloom\CMS\Modules\Architecture\Contenus\PageStatique;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Class Page
 * @package Bloom\CMS\Modules\Architecture\Model
 *
 *          Natural
 *
 * @property string  $titre
 * @property string  $meta_titre
 * @property string  $meta_description
 * @property string  $slug
 * @property string  $pathname
 * @property string  $intro
 * @property string  $multiplicity
 * @property string  $image_opengraph
 * @property int     $statut_id
 * @property boolean $is_index
 *
 *
 *           Computed
 *
 * @property Dossier $dossier
 * @property Statut  $statut
 * @property User    $cree_par
 * @property User    $deriere_modification_par
 * @property Carbon  $to_publish_at
 * @property Carbon  $to_unpublish_at
 * @property string  intro_text
 * @property Contenu $contenu
 * @property boolean is_published
 * @property boolean have_published_children
 *
 * @method static Builder|static published()
 * @method static Builder|static archived()
 */
class Page extends Model
{
    const SINGLE = '1';
    const ALL = '*';
    const MUTLI = '1-n';

    protected $guarded = [];

    protected $dates = [
        'to_publish_at',
        'to_unpublish_at',
    ];

    public static function booted()
    {
        static::saving(function (Page $page) {
            $page->pathname = '';
            if ($page->dossier) {
                $page->pathname = '/' . $page->dossier->full_pathname;
            }
            if ($page->slug) {
                $page->pathname .= '/' . $page->slug;
            }
        });
    }

    //#region Dynamique polymorphisme
    protected static $contenus = [];

    /**
     * Déclarer un nouveau model Contenu
     *
     * @param string $contenu
     */
    public static function declareContenu(string $contenu): void
    {
        if (!is_subclass_of($contenu, Contenu::class)) {
            $relation = Relation::getMorphedModel($contenu);
            if ($relation === null || !is_subclass_of($relation, Contenu::class)) {
                throw new \InvalidArgumentException("$contenu doit implémenté l'interface " . Contenu::class);
            }
        }
        if (!in_array($contenu, static::$contenus)) {
            static::$contenus[] = $contenu;
        }
    }

    public static function declareContenus(array $contenus): void
    {
        foreach ($contenus as $contenu) {
            static::declareContenu($contenu);
        }
    }

    public static function getDeclaredContenu(): array
    {
        return static::$contenus;
    }

    public function contenu()
    {
        return $this->morphTo();
    }

    public function setContenuTypeAttribute(string $value)
    {
        if (!in_array($value, static::$contenus)) {
            throw new \InvalidArgumentException('contenu_type must be a declared contenu');
        }

        $this->attributes['contenu_type'] = $value;
    }

    //#endregion

    public function setMultiplicityAttribute(string $value)
    {
        $accepted = [static::SINGLE, static::MUTLI, static::ALL];
        if (!in_array($value, $accepted)) {
            throw new \InvalidArgumentException('multiplicity should be in ' . implode(',', $accepted));
        }

        $this->attributes['multiplicity'] = $value;
    }

    public function setSlugAttribute(string $slug)
    {
        $this->attributes['slug'] = Str::slug(str_replace('/', '', $slug));
    }

    public function dossier()
    {
        return $this->belongsTo(Dossier::class);
    }

    public function statut()
    {
        return $this->belongsTo(Statut::class);
    }

    public function cree_par()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function deriere_modification_par()
    {
        return $this->belongsTo(User::class, 'last_modified_by');
    }

    public function scopePublished(Builder $query)
    {
        return $query->where('statut_id', '=', 1);
    }

    public function scopeArchived(Builder $query)
    {
        return $query->where('statut_id', '=', 0);
    }

    public function archive(): bool
    {
        $this->statut_id = Statut::ARCHIVE;

        return $this->save();
    }

    public function unarchive(): bool
    {
        $this->statut_id = Statut::EN_COURS;
        $this->to_publish_at = null;
        $this->to_unpublish_at = null;

        return $this->save();
    }

    public function publish(): bool
    {
        $this->to_publish_at = now();
        $this->statut_id = Statut::PUBLIE;

        return $this->save();
    }

    public function softDelete(): bool
    {
        $this->statut_id = Statut::SUPPRIMER;

        return $this->save();
    }

    public function unpublish(): bool
    {
        $this->statut_id = Statut::EN_COURS;
        $this->to_publish_at = null;
        $this->to_unpublish_at = null;

        return $this->save();
    }

    public function getIntroTextAttribute()
    {
        if ((string)$this->intro === '') {
            return '';
        }

        return htmlspecialchars_decode(strip_tags($this->intro), ENT_QUOTES);
    }

    public function getHasMenuAttribute()
    {
        return DB::table('menus')
                ->where('page_id', '=', $this->id)
                ->where('statut_id', '=', 1)
                ->count() > 0;
    }

    public function getHasMirroirAttribute()
    {
        return false;
    }

    public function getEditRouteAttribute()
    {
        $exception = ['contact', 'plan_du_site', 'recrutement'];

        if (in_array($this->contenu_type, $exception)) {
            return '#';
        }
        /**
         * @var Applicabilite[] $applicabilites
         */
        static $applicabilites;
        if ($applicabilites === null) {
            $applicabilites = Applicabilite::query()->whereRaw(sprintf('b_on & %d = %d', Applicabilite::ARCHI, Applicabilite::ARCHI))->get();
        }
        foreach ($applicabilites as $applicabilite) {
            if ($this->contenu instanceof $applicabilite->contenu_type) {
                return route($applicabilite->route, $this->contenu);
            }
        }

        if ($this->is_index) {
            return 'document.dispatchEvent(new CustomEvent("edit-sommaire", {detail: {route: "' . route('admin_archi_som', [Section::query()->where('page_id', '=', $this->id)->count() > 0 ? 'perso' : 'auto', $this]) . '"}}))';
        }

        return '#';
    }

    public function getIsPublishedAttribute(): bool
    {
        return $this->statut_id == Statut::PUBLIE;
    }

    public function scopeType(Builder $query, string $type)
    {
        return $query->where('contenu_type', '=', $type);
    }

    public function getHavePublishedChildrenAttribute(): bool
    {
        if (!$this->is_index) {
            return false;
        }

        foreach ($this->dossier->pages as $page) {
            if ($page->id == $this->id) continue;

            if ($page->is_published) {
                return true;
            }
        }

        return false;
    }

    public function getContenuNameAttribute(): string
    {
        if ($this->contenu && $this->contenu instanceof PageStatique) {
            return $this->contenu->display_type;
        }

        return Str::title($this->contenu_type);
    }

}