<?php


namespace Bloom\CMS\Modules\Architecture\Model;


use Bloom\CMS\Modules\Architecture\Contenus\Contenu;
use Bloom\CMS\Modules\Architecture\Contenus\PageStatique;
use Bloom\CMS\Modules\Architecture\Events\PageArchived;
use Bloom\CMS\Modules\Architecture\Events\PageDeleted;
use Bloom\CMS\Modules\Architecture\Events\PagePublished;
use Bloom\CMS\Modules\Architecture\Events\PageUnpublished;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

/**
 * Class Page
 * @package Bloom\CMS\Modules\Architecture\Model
 *
 *          Natural
 *
 * @property string  $titre
 * @property string  $meta_titre
 * @property string  $meta_description
 * @property string  $slug
 * @property string  $pathname
 * @property string  $intro
 * @property string  $multiplicity
 * @property string  $image_opengraph
 * @property int     $statut_id
 * @property boolean $is_index
 *
 *
 *           Computed
 *
 * @property Dossier $dossier
 * @property Statut  $statut
 * @property User    $cree_par
 * @property User    $deriere_modification_par
 * @property Carbon  $to_publish_at
 * @property Carbon  $to_unpublish_at
 * @property string  intro_text
 * @property Contenu $contenu
 * @property boolean is_published
 * @property boolean have_published_children
 *
 * @method static Builder|static published()
 * @method static Builder|static archived()
 */
class Page extends Model
{
    const SINGLE = '1';
    const ALL = '*';
    const MUTLI = '1-n';

    protected $guarded = [];

    protected $dates = [
        'to_publish_at',
        'to_unpublish_at',
    ];

    public static function booted()
    {
        static::saving(function (Page $page) {
            //Maj de la métadecription
            $page->changeMetaDescription();

            // Si les champs "slug" et "dossier" son mis à jour on mets à jour le pathname
            $page->updatePathname();

            if ($page->isDirty('pathname') && $page->exists) {
                // Si le pahtname a changé il faut crée des redirects
                // 301 vers la nouvelle url
                $query = Redirection::query()->where('source', '=', $page->getOriginal('pathname'));
                if ($page->is_published || $query->exists()) {
                    // Si on a une redirection de publication/dépublication qui existe on la supprime pour faire le ménage
                    if ($query->exists()) {
                        $query->delete();

                        // On crée la redirection vers le dossier parents
                        $redirection = new Redirection();
                        $redirection->setStatusChange($page);
                        if ($page->is_published) {
                            $redirection->statut_id = Statut::EN_COURS;
                        }
                        $redirection->save();
                        $redirection->bindToSystem();
                    }

                    $redirection = new Redirection();
                    $redirection->setPathChange($page);
                    $redirection->save();
                    // On bind la redirection à une redirection System
                    $redirection->bindToSystem();
                }
            }

            if ($page->isDirty('statut_id')) {
                // Si la page est dépublié on crée une redirection vers le dossier parent
                // Si une redirection existe déjà pour le couple ['source','cible'] on publie la redirection
                // Si la page deviens publié on passe la redirection en statut 2
                /**
                 * @var Redirection|null $redirection
                 */
                $redirection = Redirection::query()
                    ->where('source', '=', $page->pathname)
                    ->where('cible', '=', $page->dossier ? $page->dossier->full_pathname : '/')
                    ->first();
                if (!$page->is_published) {
                    // On crée / publie la redirection
                    if ($redirection === null) {
                        $redirection = new Redirection();
                        $redirection->setStatusChange($page);
                        if($page->statut_id == Statut::ARCHIVE) {
                            $redirection->hard = true;
                        }
                        $redirection->save();
                        $redirection->bindToSystem();
                    }
                    $redirection->statut_id = Statut::PUBLIE;
                    $redirection->save();
                } else {
                    // On dépublie la redirection si elle existe
                    if ($redirection !== null) {
                        $redirection->statut_id = Statut::EN_COURS;
                        $redirection->save();
                    }
                }

                if ($page->contenu) {
                    // On trigger les events de publication / dépublication de page
                    switch ((int)$page->statut_id) {
                        case Statut::SUPPRIMER:
                            Event::dispatch(new PageDeleted($page));
                            break;
                        case Statut::ARCHIVE:
                            Event::dispatch(new PageArchived($page));
                            break;
                        case Statut::EN_COURS:
                            if ($page->getOriginal('statut_id') == Statut::PUBLIE) {
                                Event::dispatch(new PageUnpublished($page));
                            }
                            break;
                        case Statut::PUBLIE:
                            Event::dispatch(new PagePublished($page));
                    }
                }
            }
        });
    }

    public function updatePathname()
    {
        $this->pathname = '';
        if ($this->dossier) {
            $this->pathname = '/' . $this->dossier->full_pathname;
        }
        if ($this->slug) {
            $this->pathname .= '/' . $this->slug;
        }

        $query = $this->newQuery()->where('pathname', '=', $this->pathname);
        if ($this->exists) {
            $query->where('id', '!=', $this->getKey());
        }
        Log::debug('Checking duplicate pathname', ["pathname" => $this->pathname]);
        if (
            // Si on a un doublons d'url
            $query->count() > 0 ||
            // Ou si une redirection en 301 existe déjà sur l'url actuel
            // Cas d'un déplacement de page, le pathname d'origine n'existe plus dans la table page
            // mais il ne faut quand même pas crée de page portant l'ancien pathname
            Redirection::query()
                ->where('statut_id', '=', Statut::PUBLIE)
                ->where('source', '=', $this->pathname)
                ->where('hard', '=', 1)
                ->exists()
        ) {
            $this->uniquePathname($this->pathname);
        }
    }

    protected function uniquePathname(string $original)
    {
        $query = $this->newQuery()->whereRaw('pathname REGEXP "^' . $original . '(-[1-9]+)*$"');
        if ($this->exists) {
            $query->where('id', '!=', $this->getKey());
        }
        $query2 = Redirection::query()
            ->where('statut_id', '=', Statut::PUBLIE)
            ->where('source', '=', $this->pathname)
            ->where('hard', '=', 1);

        $query3 = Miroir::query()->where('pathname', '=', $this->pathname);

        $this->pathname .= "-" . ($query->count() + $query2->count() + $query3->count());
        $this->slug .= "-" . ($query->count() + $query2->count() + $query3->count());
        Log::debug('Setting unique pathname', ["pathname" => $this->pathname]);
    }

    protected function changeMetaDescription() {
        // Position de la première ponctuation (‘.’, ‘!’ ou ‘?’) dans l'intro trouvée après 100 caractères
        $search_char = ['.', '!', '?'];
        $pos = false;
        $array_pos = [];
        if(Str::length($this->intro_text) >= 100) {
            foreach ($search_char as $char) {
                $pos = strpos($this->intro_text, $char, 100);
                if(is_numeric($pos)) array_push($array_pos, $pos);
            }

            $pos = min($array_pos);
        }

        // Modification du champ meta_description
        if (!$pos)
            $this->meta_description = html_entity_decode($this->intro_text, ENT_QUOTES, 'UTF-8');
        else
            $this->meta_description = html_entity_decode(substr($this->intro_text, 0, $pos + 1), ENT_QUOTES, 'UTF-8');
    }

    //#region Dynamique polymorphisme
    protected static $contenus = [];

    /**
     * Déclarer un nouveau model Contenu
     *
     * @param string $contenu
     */
    public static function declareContenu(string $contenu): void
    {
        if (!is_subclass_of($contenu, Contenu::class)) {
            $relation = Relation::getMorphedModel($contenu);
            if ($relation === null || !is_subclass_of($relation, Contenu::class)) {
                throw new \InvalidArgumentException("$contenu doit implémenté l'interface " . Contenu::class);
            }
        }
        if (!in_array($contenu, static::$contenus)) {
            static::$contenus[] = $contenu;
        }
    }

    public static function declareContenus(array $contenus): void
    {
        foreach ($contenus as $contenu) {
            static::declareContenu($contenu);
        }
    }

    public static function getDeclaredContenu(): array
    {
        return static::$contenus;
    }

    public function contenu()
    {
        return $this->morphTo();
    }

    public function setContenuTypeAttribute(string $value)
    {
        if (!in_array($value, static::$contenus)) {
            throw new \InvalidArgumentException('contenu_type must be a declared contenu');
        }

        $this->attributes['contenu_type'] = $value;
    }

    //#endregion

    public function setMultiplicityAttribute(string $value)
    {
        $accepted = [static::SINGLE, static::MUTLI, static::ALL];
        if (!in_array($value, $accepted)) {
            throw new \InvalidArgumentException('multiplicity should be in ' . implode(',', $accepted));
        }

        $this->attributes['multiplicity'] = $value;
    }

    public function setSlugAttribute(string $slug)
    {
        $this->attributes['slug'] = Str::slug(str_replace('/', '', $slug));
    }

    public function miroir()
    {
        return $this->hasMany(Miroir::class);
    }

    public function miroirNotArchived()
    {
        return $this->hasMany(Miroir::class)->where('statut_id', '!=', Statut::ARCHIVE);
    }

    public function dossier()
    {
        return $this->belongsTo(Dossier::class);
    }

    public function statut()
    {
        return $this->belongsTo(Statut::class);
    }

    public function cree_par()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function deriere_modification_par()
    {
        return $this->belongsTo(User::class, 'last_modified_by');
    }

    public function scopePublished(Builder $query)
    {
        return $query->where('statut_id', '=', 1);
    }

    public function scopeArchived(Builder $query)
    {
        return $query->where('statut_id', '=', 0);
    }

    public function archive(): bool
    {
        $this->statut_id = Statut::ARCHIVE;

        return $this->save();
    }

    public function unarchive(): bool
    {
        $this->statut_id = Statut::EN_COURS;
        $this->to_publish_at = null;
        $this->to_unpublish_at = null;

        return $this->save();
    }

    public function publish(): bool
    {
        $this->to_publish_at = now();
        $this->statut_id = Statut::PUBLIE;

        return $this->save();
    }

    public function softDelete(): bool
    {
        $this->statut_id = Statut::SUPPRIMER;

        return $this->save();
    }

    public function unpublish(): bool
    {
        $this->statut_id = Statut::EN_COURS;
        $this->to_publish_at = null;
        $this->to_unpublish_at = null;

        return $this->save();
    }

    public function getIntroTextAttribute()
    {
        if ((string)$this->intro === '') {
            return '';
        }

        return htmlspecialchars_decode(strip_tags($this->intro), ENT_QUOTES);
    }

    public function getHasMenuAttribute()
    {
        return DB::table('menus')
                ->where('page_id', '=', $this->id)
                ->where('statut_id', '=', 1)
                ->count() > 0;
    }

    public function getMenuAttribute()
    {
        if ($this->has_menu) {
            return DB::table('menus')
                ->where('page_id', '=', $this->id)
                ->where('statut_id', '=', 1)
                ->first();
        }

        return null;
    }

    public function getHasMiroirsAttribute()
    {
        return DB::table('page_dossiers')->where('page_id', '=', $this->id)->exists();
    }

    public function getEditRouteAttribute()
    {
        $exception = ['contact', 'plan_du_site', 'recrutement'];

        if (in_array($this->contenu_type, $exception)) {
            return '#';
        }
        /**
         * @var Applicabilite[] $applicabilites
         */
        static $applicabilites;
        if ($applicabilites === null) {
            $applicabilites = Applicabilite::query()->whereRaw(sprintf('b_on & %d = %d', Applicabilite::ARCHI, Applicabilite::ARCHI))->get();
        }
        foreach ($applicabilites as $applicabilite) {
            if ($this->contenu instanceof $applicabilite->contenu_type) {
                return route($applicabilite->route, $this->contenu);
            }
        }

        if ($this->is_index) {
            return 'document.dispatchEvent(new CustomEvent("edit-sommaire", {detail: {route: "' . route('admin_archi_som', [Section::query()->where('page_id', '=', $this->id)->count() > 0 ? 'perso' : 'auto', $this]) . '"}}))';
        }

        return '#';
    }

    public function getIsPublishedAttribute(): bool
    {
        return $this->statut_id == Statut::PUBLIE;
    }

    public function scopeType(Builder $query, string $type)
    {
        return $query->where('contenu_type', '=', $type);
    }

    public function getHavePublishedChildrenAttribute(): bool
    {
        if (!$this->is_index) {
            return false;
        }

        foreach ($this->dossier->pages as $page) {
            if ($page->id == $this->id) continue;

            if ($page->is_published) {
                return true;
            }
        }

        return false;
    }

    public function getContenuNameAttribute(): string
    {
        if ($this->contenu && $this->contenu instanceof PageStatique) {
            return $this->contenu->display_type;
        }

        return Str::title($this->contenu_type);
    }

}