<?php

namespace Bloom\CMS\Modules\Alert\Http\Controllers;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Workflow;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Lien;
use Bloom\CMS\Core\Http\Miroir;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Alert\Model\Alert;
use Bloom\CMS\Modules\Alert\Model\Node;
use Carbon\Carbon;
use Illuminate\Routing\Controller;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Traversable;

class AdminController extends Controller
{
    /**
     * @param Page          $page
     * @param Miroir|null   $pageDossier
     *
     * @return Node
     */
    public function buildPage(Page $page, Miroir $pageDossier = null): Node
    {
        $node = new Node();
        if ($pageDossier) {
            $node->id = $pageDossier->id;
            $node->type = "miroir";
        } else {
            $node->id = $page->id;
            $node->type = "pages";
        }
        $node->title = $page->titre;

        return $node;
    }

    /**
     * @param Dossier $dossier
     *
     * @return Node
     */
    public function buildTree(Dossier $dossier): Node
    {
        $node = new Node();
        $node->type = "pages";

        $pages = $dossier->pages()
            ->where('statut_id', '=', Statut::PUBLIE)
            ->get();
        foreach ($pages as $page) {
            if ($page->is_index) {
                $node->title = $page->titre;
                $node->id = $page->id;
            } else {
                $node->children[] = $this->buildPage($page, null);
            }
        }

        $children = $dossier->childs;
        foreach ($children as $child) {
            $node->children[] = $this->buildTree($child);
        }

        $pagesDossiers = $dossier->miroirs()->where('statut_id', '=', Statut::PUBLIE)->get();
        foreach ($pagesDossiers as $pagesDossier) {
            $page = Page::published()
                ->where('id', '=', $pagesDossier->page_id)
                ->first();
            if ((int)$pagesDossier->page->statut_id === Statut::PUBLIE) {
                $node->children[] = $this->buildPage($page, $pagesDossier);
            }
        }

        return $node;
    }

    /**
     * @param Traversable|Dossier[] $dossiers
     * @param int                   $level
     *
     * @return Collection
     */
    public function sortDossiers(Traversable $dossiers, int $level): Collection
    {
        $tab = collect();

        foreach ($dossiers as $dossier) {
            $dossier->level = $level;
            $tab[] = $dossier;
            if ($dossier->childs) {
                $tab = $tab->concat($this->sortDossiers($dossier->childs, $level + 1));
            }
        }

        return $tab;
    }


    public function edit(Alert $alert)
    {
        $node = new Node();

        $pages = Page::published()->whereNull('dossier_id')->get();
        $tableDossiers = Dossier::query()->whereNull('parent_id')->get();

        foreach ($pages as $page) {
            $node->children[] = $this->buildPage($page, null);
        }
        foreach ($tableDossiers as $dossier) {
            $node->children[] = $this->buildTree($dossier);
        }

        $dossiers = $this->sortDossiers($tableDossiers, 0);

        $page = $alert->page;
        if ($page === null) {
            $page = new Page();
        }

        return view('Alert::admin.edit', compact('alert', 'dossiers', 'node', 'page'));
    }

    /**
     * Permet de fill les parametre générique d'une page avec le formulaire
     *
     * @param Page $page
     */
    protected function fillPage(Page $page): void
    {
        $page->titre = request('titre');
        $page->intro = request('intro');
        $page->dossier_id = request('dossier_id');
        $page->to_publish_at = request('date_pub');
        $page->to_unpublish_at = request('date_depub');
        $page->mentions = request('mentions');
    }

    /**
     * @param Alert $alert
     * @param int   $statut
     *
     * @return int
     */
    public function createAlert(Alert $alert, int $statut): int
    {
        // On crée l'alerte
        $alert->contenu = request('contenu');
        $alert->one_time = request('one_time');
        $alert->save();

        // On crée la page
        $page = new Page();
        $page->meta_titre = '';
        $page->meta_description = '';
        $page->created_by = Auth::user()->getAuthIdentifier();
        $page->last_modified_by = Auth::user()->getAuthIdentifier();
        $page->multiplicity = Page::SINGLE;
        $this->fillPage($page);

        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = 2;
        }
        else if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isCurrentDay()) {
            $statut = 1;
            $this->checkIfNewAlertIsPublished($alert);
        }
        if (request('statut_id') == 1) {
            $statut = 1;
            $page->to_publish_at = now();
            $this->checkIfNewAlertIsPublished($alert);
        }

        $page->statut_id = $statut;
        // On attache la page et l'alerte
        $page->contenu()->associate($alert);
        $page->save();

        return $statut;
    }

    /**
     * @param Alert $alert
     * @param int   $statut
     *
     * @return int
     */
    public function editAlert(Alert $alert, int $statut): int
    {
        /**
         * @var Page $page
         */
        $page = $alert->page;

        $alert->contenu = request('contenu');
        $alert->one_time = request('one_time');
        if ($alert->isDirty()) {
            // On mets à jour la page
            $page->last_modified_by = Auth::user()->getAuthIdentifier();
            $alert->save();
        }

        $this->fillPage($page);

        if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
            $statut = 2;
        }
        if (request('statut_id') == 1) {
            $this->checkIfNewAlertIsPublished($alert);
            $statut = 1;
            $page->to_publish_at = now();
        }
        $page->statut_id = $statut;

        if ($page->isDirty()) {
            $page->last_modified_by = Auth::user()->getAuthIdentifier();
            $page->save();
            $alert->touch();
        }

        return $statut;
    }

    /**
     * @param Alert $alert
     */
    public function checkIfNewAlertIsPublished(Alert $alert)
    {
        $alertPublished = Alert::published()->first();

       if ($alertPublished && $alert->id != $alertPublished->id) {
           $page = $alertPublished->page;
           $page->to_unpublish_at = now();
           $page->statut_id = Statut::ARCHIVE;
           $alertPublished->page()->save($page);
       }
    }

    /**
     * @param Alert $alert
     */
    public function getCTA(Alert $alert)
    {
        for ($i = 1; $i <= 2; $i++) {
            $lien = 'lien_' . $i;
            $method = 'cta_' . $i;

            if (request($lien) && request('link_choice_lien_' . $i)) {
                /**
                 * @var Lien $newLien
                 */
                $newLien = $alert->$lien != null ? $alert->$method : new Lien();
                $newLien->libelle = request('lien_' . $i . '_libelle');

                if (request('link_choice_lien_' . $i)) {

                    $page_id = request('lien_' . $i . '_page_id');
                    $type = 'pages';
                    if ($page_id !== null) {
                        list($page_id, $type) = explode(';', $page_id);
                    }
                    if ($type === 'pages') {
                        $newLien->page_id = $page_id;
                        $newLien->miroir_id = null;
                    } else {
                        $newLien->miroir_id = $page_id;
                        $newLien->page_id = null;
                    }
                    $newLien->href = null;
                }

                if (request('lien_' . $i . '_href')) {
                    $parsed_link = parse_url(request('lien_' . $i . '_href'));
                    $newLien->href = empty($parsed_link['scheme']) ? 'http://' . request('lien_' . $i . '_href') : $newLien->href = request('lien_' . $i . '_href');
                    $newLien->page_id = null;
                }

                $newLien->save();

                if (!$alert->$method) $alert->$method()->associate($newLien);
            } else {
                if ($alert->$method) $alert->$method()->delete();
                $alert->$lien = null;
            }
        }
        $alert->save();
    }

    public function save(Alert $alert)
    {
        $statut = 2;

        if ($alert->page !== null) {
            $statut = $alert->page->statut_id ?: request('statut_id', 2);
        }
        $statut = $alert->exists ? $this->editAlert($alert, $statut) : $this->createAlert($alert, $statut);

        $this->getCTA($alert);

        if ($statut == 2) {
            // on reset la relation page pour la vérification du publish at
            $alert->unsetRelation('page');
            if ($alert->page->to_publish_at) {
                $workflow = Workflow::query()->where('code', '=', 'ALERT_PROGRAMMED')->first();
                return redirect()->route('admin_alert_index', $workflow);
            }
            $workflow = Workflow::query()->where('code', '=', 'ALERT_EN_COURS')->first();

            return redirect()->route('admin_alert_index', $workflow);
        }

        return redirect()->route('admin_alert_index');
    }

    public function preview(Alert $alert)
    {
        return $alert->getHandler()->single($alert);
    }

    public function publish(Alert $alert)
    {
        /**
         * @var Page $page
         */
        $page = $alert->page;
        $page->publish();

        return redirect()->back();
    }

    public function unpublish(Alert $alert)
    {
        /**
         * @var Page $page
         */
        $page = $alert->page;
        $page->unpublish();

        return redirect()->back();
    }

    public function archive(Alert $alert)
    {
        /**
         * @var Page $page
         */
        $page = $alert->page;
        $page->archive();

        return redirect()->back();
    }

    public function duplicate(Alert $alert)
    {
        /**
         * @var Page $page
         */
        $page = $alert->page;
        $page2 = $page->replicate();
        $page2->titre .= ' - Copie';
        $page2->slug .= '-copie';
        $page2->pathname .= '-copie';
        // L’entité dupliquée prend le statut ‘en cours’
        $page2->statut_id = Statut::EN_COURS;
        $page2->to_publish_at = null;
        $page2->to_unpublish_at = null;
        $page2->save();

        $alert2 = $alert->replicate();
        $alert2->save();
        $page2->contenu()->associate($alert2);
        $page2->save();
        // Ouverture de l’onglet/workflow ‘en cours’
        $workflow = Workflow::query()->where('code', '=', 'ALERT_EN_COURS')->first();

        return redirect()->route('admin_alert_index', $workflow);
    }

    public function delete(Alert $alert)
    {
        /**
         * @var Page $page
         */
        $page = $alert->page;
        $page->softDelete();

        return redirect()->back();
    }
}
