<?php

namespace Bloom\CMS\Modules\Alert\Model;

use Bloom\CMS\Core\Contenus\Contenu;
use Bloom\CMS\Core\Contenus\ContenuHandler;
use Bloom\CMS\Core\Contenus\IsContenu;
use Bloom\CMS\Core\Http\Lien;
use Bloom\CMS\Core\Http\Page;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Log;

/**
 * Class Alert
 *
 * @package Bloom\CMS\Modules\Alert\Model
 *
 * @property int        id
 * @property string     contenu
 * @property string     lien_1
 * @property string     lien_2
 * @property boolean    one_time
 * @property Page       page
 *
 * @method static Builder|static published()
 * @method static Builder|static archived()
 * @method static Builder|static programmed()
 * @method static Builder|static notProgrammed()
 * @method static Builder|static programmedFor(Carbon $date)
 * @method static Builder|static status(int $status)
 */
class Alert extends Model implements Contenu
{
    use IsContenu;

    public function getHandler(): ContenuHandler
    {
        return new class implements ContenuHandler {

            public function single(Contenu $contenu): Response
            {
                /**
                 * @var Page $page
                 */
                $page = $contenu->page;
                $breadcrumbElements = [["name" => $page->titre, 'url' => '']];
                if ($page->dossier) {
                    $dossier = $page->dossier;
                    while ($dossier !== null) {
                        $breadcrumbElements[] = ['name' => $dossier->name, 'url' => $dossier->full_pathname];
                        $dossier = $dossier->parent;
                    }
                }

                $breadcrumbElements = array_reverse($breadcrumbElements);

                return response(
                    view(
                        'Alert::front.index',
                        ['alert' => $contenu, 'page' => $page, 'breadcrumbElements' => $breadcrumbElements]
                    )
                );
            }

            public function multiple(Collection $contenus): Response
            {
                Log::warning('multiple alert not unhandled', ['contenu' => $contenus]);

                return \response();
            }

            public function ensemble()
            {
                Log::warning('ensemble alert not unhandled');
            }
        };
    }

    public function cta_1(): BelongsTo
    {
        return $this->belongsTo(Lien::class, 'lien_1');
    }

    public function cta_2(): BelongsTo
    {
        return $this->belongsTo(Lien::class, 'lien_2');
    }


    public function scopePublished(Builder $query)
    {
        return $this->scopeStatus($query, 1);
    }


    public function scopeArchived(Builder $query)
    {
        return $this->scopeStatus($query, 0);
    }

    public function scopeStatus(Builder $query, int $status)
    {
        return $query->whereHas(
            'page',
            function (Builder $query) use ($status) {
                return $query->where('statut_id', '=', $status);
            }
        );
    }

    public function scopeProgrammed(Builder $query)
    {
        return $query->whereHas(
            'page',
            function (Builder $query) {
                return $query->whereNotNull('to_publish_at')
                    ->where('statut_id', '=', 2);
            }
        );
    }

    public function scopeNotProgrammed(Builder $query)
    {
        return $query->whereHas(
            'page',
            function (Builder $query) {
                return $query->whereNull('to_publish_at');
            }
        );
    }
}
