<?php

namespace Bloom\CMS\Modules\Agences\Model;

use Bloom\CMS\Modules\Architecture\Helpers\IsContenu;
use Bloom\CMS\Modules\Architecture\Http\Controllers\ContenuHandler;
use Bloom\CMS\Modules\Architecture\Contenus\Contenu;
use Bloom\CMS\Modules\Architecture\Model\Page;
use Bloom\CMS\Modules\Architecture\Model\Statut;
use Illuminate\Http\Response;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

/**
 * Class Agence
 * @package Bloom\Bloom\CMS\Modules\Agences\Model
 *
 * @property string             $contenu
 * @property Page[]|Collection  page
 *
 *
 * @property integer    $id
 * @property string     $adresse
 * @property string     $cp
 * @property string     $ville
 * @property integer    $rank
 * @property string     $position
 * @property string     $horaires
 * @property integer    $statut_id
 * @property Statut     $status
 *
 * @method static Builder|static published()
 * @method static Builder|static archived()
 * @method static Builder|static status(int $status)
 *
 */
class Agence extends Model implements Contenu
{

    use IsContenu;

    protected $appends = ['point'];

    protected $hidden = ['position'];

    protected $casts = [
        'horaires' => 'array',
    ];

    public function getHandler(): ContenuHandler
    {
        return new class implements ContenuHandler {

            public function single(Contenu $contenu): Response
            {
                /**
                 * @var Page $page
                 */
                $page = $contenu->page;
                $breadcrumbElements = [["name" => $page->titre, 'url' => '']];
                if ($page->dossier) {
                    $dossier = $page->dossier;
                    while ($dossier !== null) {
                        $breadcrumbElements[] = ['name' => $dossier->name, 'url' => $dossier->full_pathname];
                        $dossier = $dossier->parent;
                    }
                }

                $breadcrumbElements = array_reverse($breadcrumbElements);

                return response(view('Agences::front.index', ['agence' => $contenu, 'page' => $page, 'breadcrumbElements' => $breadcrumbElements]));
            }

            public function multiple(Collection $contenus): Response
            {
                // TODO: Implement multiple() method.
            }

            public function ensemble()
            {
                /**
                 * @var Collection $agences
                 */
                $agences = Agence::published()->join('pages', function ($join) {
                    $join->on('agences.id', '=', 'pages.contenu_id')
                        ->where('pages.contenu_type', '=', 'agences');
                })
                    ->get();

                $breadcrumbElements = [["name" => 'Agences', 'url' => '']];

                $breadcrumbElements = array_reverse($breadcrumbElements);

                return response(view('Agences::front.listing', compact('agences', 'breadcrumbElements')));
            }
        };
    }

    public function getPointAttribute()
    {
        $position = \DB::table($this->getTable())->selectRaw('ST_X(position) as lat, ST_Y(position) as lng')->where('id', '=', $this->id)->first();

        return $position;
    }

    public function equipes()
    {
        return $this->hasMany(Equipe::class)->orderBy('rank');
    }


    public function scopeStatus(Builder $query, int $status)
    {
        return $query->whereHas('page', function (Builder $query) use ($status) {
            return $query->where('statut_id', '=', $status);
        });
    }

    public function scopePublished(Builder $query)
    {
        return $this->scopeStatus($query, Statut::PUBLIE);
    }

    public function scopeArchived(Builder $query)
    {
        return $this->scopeStatus($query, Statut::ARCHIVE);
    }
}
