<?php
/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Agences\Http\Controllers;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Actions;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Agences\CmsAgencesServiceProvider;
use Bloom\CMS\Modules\Agences\Model\Agence;
use Bloom\CMS\Modules\Agences\Model\Equipe;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

/**
 * Class AdminController
 * @package Bloom\CMS\Modules\Agences\Http\Controllers
 */
class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiées',
                    'href'   => route('admin_agence_index'),
                    'number' => Agence::published()->count(),
                    'active' => request()->route()->getName() === 'admin_agence_index',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_agence_encours', ['orderBy' =>'page.updated_at', 'order' => 'desc']),
                    'number' => Agence::status(Statut::EN_COURS)->count(),
                    'active' => request()->route()->getName() === 'admin_agence_encours',
                ],
                [
                    'title'  => 'Archivés',
                    'href'   => route('admin_agence_archived', ['orderBy' =>'page.updated_at', 'order' => 'desc']),
                    'number' => Agence::archived()->count(),
                    'active' => request()->route()->getName() === 'admin_Agence_archived',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(bool $with_sub = true): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_agence_edit')
            ->addNewTabAction('Prévisualiser', 'admin_agence_preview')
            ->addAction('Publier', 'admin_agence_publish');
        if ($with_sub) {
            $sub_actions = new Actions('sub_actions');
            $sub_actions
                ->addAction('Archiver', 'admin_agence_archive');
        }

        return array_merge($actions->toArray(), isset($sub_actions) ? $sub_actions->toArray() : []);
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');
        $sort = function (Agence $agence) {
            return Arr::get(Arr::dot($agence->toArray()), request('orderBy', 'rank'));
        };

        return [
            'agences' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order'   => [
                'item'  => request('orderBy', 'rank'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions();

        // On remplace la 2ieme action du menu
        $actions['actions'][2] = [
            'name' => 'Dépublier', 'href' => 'admin_agence_unpublish'
        ];

        $data = array_merge($this->getSorted(Agence::published()->orderBy('rank')->get()), $actions);

        return view('Agence::admin.listing', $this->getViewData($data));
    }


    public function encours()
    {
        $data = array_merge($this->getSorted(Agence::status(Statut::EN_COURS)->get()), $this->getDefaultActions());

        return view('Agence::admin.listing', $this->getViewData($data));
    }

    public function archived()
    {
        $data = array_merge(
            $this->getSorted(Agence::archived()->get()),
            (new Actions())
                ->addAction('Désarchiver', 'admin_agence_unpublish')->toArray()
        );

        return view('Agence::admin.listing', $this->getViewData($data));
    }

    public function edit(Agence $agence)
    {
        $dossiers = Dossier::all();
        $page = $agence->page;
        if (request('dossier_id')) {
            if ($page === null) {
                $page = new Page();
            }
            $page->dossier_id = request('dossier_id');
        }

        return view('Agence::admin.edit', compact('agence', 'dossiers', 'page'));
    }

    public function save(Agence $agence)
    {
        $user_id = Auth::user()->getAuthIdentifier();
        $dossier = Dossier::where('name', '=', CmsAgencesServiceProvider::DOSSIER)->first();

        if (request('statut_id')) {
            $statut = request('statut_id');
        } elseif ($agence->page !== null) {
            $statut = $agence->page->statut_id ?: Statut::EN_COURS;
        } else {
            $statut = Statut::EN_COURS;
        }

        // Gestion des champs spécifiques aux agences
        $agence->adresse = request('adresse');
        $agence->cp = request('cp');
        $agence->novadial = request('novadial', '');
        $agence->ville = request('ville');
        $agence->position = DB::raw("ST_SRID('POINT(" . request(('point_lat')) . " " . request('point_lng') . "),4326')");


        //Gestion des horaires
        $days = ['Lundi', 'Mardi', 'Mercredi', 'Jeudi', 'Vendredi', 'Samedi'];
        $hour = ['debut_matin', 'fin_matin', 'debut_am', 'fin_am',];
        $horaires = [];

        for ($i = 1; $i <= 6; $i++) {
            for ($j = 1; $j <= 4; $j++) {
                $horaires[ $days[ $i - 1 ] ][ $hour[ $j - 1 ] ] = request('hour0' . ($i) . '_0' . $j) ?: '';
            }
        }

        $agence->horaires = array($horaires);


        if (!$agence->exists) {
            // Nouveau

            // On crée l'agence
            $agence->rank = Agence::all()->count();
            $agence->save();

            // On crée la page
            $page = new Page();
            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->meta_titre = '';
            $page->meta_description = '';
            $page->created_by = $user_id;
            $page->last_modified_by = $user_id;
            $page->intro = request('intro');
            $page->dossier()->associate($dossier);
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->multiplicity = Page::SINGLE;
            $page->image_opengraph = request('vignette');
            $page->mentions = request('mentions');

            if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
                $statut = Statut::EN_COURS;
            }
            if (request('statut_id') == Statut::EN_COURS) {
                $statut = Statut::PUBLIE;
                $page->to_publish_at = now();
            }

            $page->statut_id = $statut;
            // On attache la page et l'article
            $page->contenu()->associate($agence);
            $page->save();

        } else {
            // Edition

            /**
             * @var Page $page
             */
            $page = $agence->page;

            if ($agence->isDirty()) {
                // On mets à jour la page
                $page->last_modified_by = $user_id;
                $agence->save();
            }

            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->intro = request('intro');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->image_opengraph = request('vignette');
            $page->mentions = request('mentions');

            if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
                $statut = Statut::EN_COURS;
            }
            if (request('statut_id') == Statut::PUBLIE) {
                $statut = Statut::PUBLIE;
                $page->to_publish_at = now();
            }
            $page->statut_id = $statut;

            if ($page->isDirty()) {
                $page->last_modified_by = $user_id;
                $page->save();
                $agence->touch();
            }
        }

        //Gestion des équipes
        $equipes = json_decode(request('equipes'));

        if ($equipes) {
            foreach ($equipes as $equipe) {
                $team = $equipe->id == '' ? new Equipe() : $agence->equipes()->find($equipe->id);
                if ($equipe->agence_id == '') $team->agence_id = $agence->id;
                $team->nom = $equipe->nom;
                $team->prenom = $equipe->prenom;
                $team->fonction = $equipe->fonction;
//       $team->rank = $equipe->id != null ? $equipe->rank : Equipe::where('agence_id', '=', $agence->id)->count() + 1;
                $team->rank = $equipe->rank;
                $team->image = $equipe->image;
                $team->save();
            }
        }


        if ($statut == Statut::EN_COURS) {
            return redirect()->route('admin_agence_encours');
        }

        if ($statut == Statut::ARCHIVE) {
            return redirect()->route('admin_agence_archived');
        }

        return redirect()->route('admin_agence_index');
    }

    public function preview(Agence $agence)
    {
        return $agence->getHandler()->single($agence);
    }

    public function archive(Agence $agence)
    {
        /**
         * @var Page $page
         */
        $page = $agence->page;
        $page->archive();

        return redirect()->back();
    }

    public function publish(Agence $agence)
    {
        /**
         * @var Page $page
         */
        $page = $agence->page;
        $page->publish();

        return redirect()->back();
    }

    public function unpublish(Agence $agence)
    {
        /**
         * @var Page $page
         */
        $page = $agence->page;
        $page->unpublish();

        return redirect()->back();
    }

    public function delete(Agence $agence)
    {
        /**
         * @var Page $page
         */
        $page = $agence->page;
        $page->softDelete();

        return redirect()->back();
    }

    public function reorder()
    {
        $ordre = request('ordre', []);

        $agences = Agence::published()->get();

        foreach ($agences as $agence) {
            /**
             * @var Agence $agence
             */
            $agence->rank = array_search($agence->id, $ordre, false);
            $agence->save();
        }
    }

    // Gestion des Equipes
    public function deleteEquipe($equipe_id)
    {
        Equipe::find($equipe_id)->delete();
    }
}
