<?php
/**
 * Created by : loic
 * Created at : 22/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Tests\Feature;


use Bloom\CMS\Modules\Architecture\Model\Page;
use Bloom\CMS\Modules\Actu\Model\Actu;
use Faker\Factory;
use Illuminate\Foundation\Auth\User;
use Illuminate\Http\Testing\File;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\Test;

class ActuTest extends Test
{

    public function setUp(): void
    {
        parent::setUp();
        Storage::fake();
    }

    public function test_index_actus_should_show_actus()
    {
        $response = $this->actingAs(new User())->get(route('admin_actu_index'));

        $response->isOk();
        $response->assertViewIs('Actu::admin.listing');
        $response->assertViewHas('actus');
    }

    public function test_edit_actu_should_show_form()
    {
        $response = $this->actingAs(new User())->get(route('admin_actu_edit'));

        $response->isOk();
        $response->assertViewIs('Actu::admin.edit');
    }

    public function test_edit_actu_should_show_form_with_actu()
    {
        /**
         * @var Actu $actu
         */
        $actu = factory(Actu::class)->create();
        $actu->page = factory(Page::class)->create();

        $response = $this->actingAs(new User())->get(route('admin_actu_edit', $actu));

        $response->isOk();
        $response->assertViewIs('Actu::admin.edit');

        /**
         * @var Actu $actual
         */
        $actual = $response->viewData('actu');

        $this->assertInstanceOf(Actu::class, $actual);
        $this->assertTrue($actual->exists);
    }

    public function form_provider()
    {
        $this->setUp();
        $faker = Factory::create();
        $array = [];
        for ($i = 0; $i < 10; $i++) {
            $array[] = [
                [ // Form
                    'titre'      => $faker->text(50),
                    'slug'       => $faker->slug,
                    'vignette'      => UploadedFile::fake()->image('vignette.jpg'),
                    // Les html ont un \n qui est trim par Laravel, donc on le supprime directement dans la source
                    'contenu'    => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'intro'      => preg_replace('/\n$/', '', $faker->randomHtml()),
                    'dossier_id' => null,
                    'statut_id'  => 1
                ]
            ];
        }

        return $array;
    }

    /**
     * @param array $form
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_save_actu(array $form)
    {

        $user = new User();
        $user->id = 1;
        $response = $this->actingAs($user)->post(route('admin_actu_save'), $form);

        /**
         * @var File $image
         */
        $image = $form['vignette'];

        $response->isOk();

        $this->assertDatabaseHas('actus', [
            'contenu' => $form['contenu']
        ]);
        $this->assertDatabaseHas('pages', [
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'intro'           => $form['intro'],
            'image_opengraph' => Storage::disk('images')->url($image->getFilename())
        ]);

        $actu = Actu::first();

        $this->assertDatabaseHas('page_contenus', [
            'contenu_id' => $actu->id,
            'page_id'    => $actu->page->id
        ]);

        Storage::disk('images')->assertExists($image->getFilename());
    }


    /**
     * @param array $form
     *
     * @depends      test_post_edit_should_save_actu
     *
     * @dataProvider form_provider
     */
    public function test_post_edit_should_edit_actu(array $form)
    {

        $user = new User();
        $user->id = 1;

        // Sa permet de crée l'actu / page necessaire sans ce prendre la tête a tout réecrire
        $this->test_post_edit_should_save_actu($form);

        /**
         * @var Actu $actu
         * @var Page    $page
         */
        $actu = Actu::first();
        $page = $actu->page;

        // On modifie des trucs
        $form['titre'] = 'test des truc';
        $form['slug'] = 'test-des-truc';
        // On fait comme si l'image n'était pas touché
        $form['image'] = $page->image_opengraph;


        $response = $this->actingAs($user)->post(route('admin_actu_save', $actu), $form);
        $response->isOk();

        $this->assertDatabaseHas('actus', [
            'contenu' => $form['contenu']
        ]);

        $this->assertDatabaseHas('pages', [
            'id'              => $page->id,
            'titre'           => $form['titre'],
            'slug'            => $form['slug'],
            'image_opengraph' => $form['image']
        ]);

    }
}