<?php
/**
 * Created by : loic
 * Created at : 21/04/2020
 *
 * In : IntelliJ IDEA
 */

namespace Bloom\CMS\Modules\Actu\Http\Controllers;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Helpers\Actions;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Modules\Actu\Model\Actu;
use Bloom\CMS\Modules\Actu\Model\Categorie;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{

    public function getSubMenu(): array
    {
        return [
            'sub_menu_items' => [
                [
                    'title'  => 'Publiés',
                    'href'   => route('admin_actu_index'),
                    'number' => Actu::published()->count(),
                    'active' => request()->route()->getName() === 'admin_actu_index',
                ],
                [
                    'title'  => 'Programmés',
                    'href'   => route('admin_actu_programmes'),
                    'number' => Actu::programmed()->count(),
                    'active' => request()->route()->getName() === 'admin_actu_programmes',
                ],
                [
                    'title'  => 'En cours de rédaction',
                    'href'   => route('admin_actu_encours'),
                    'number' => Actu::status(2)->notProgrammed()->count(),
                    'active' => request()->route()->getName() === 'admin_actu_encours',
                ],
            ]
        ];
    }

    public function getViewData(array $data): array
    {
        return array_merge($data, $this->getSubMenu());
    }

    public function getDefaultActions(bool $with_sub = true): array
    {
        $actions = new Actions();
        $actions
            ->addAction('Modifier', 'admin_actu_edit')
            ->addNewTabAction('Prévisualiser', 'admin_actu_preview');
//            ->addAction('Nouvelle version', '');
        if ($with_sub) {
            $sub_actions = new Actions('sub_actions');
            $sub_actions
                ->addAction('Publier', 'admin_actu_publish', 'bloomicon-admin-arrow-up-right')
                ->addAction('Supprimer', 'admin_actu_delete', 'bloomicon-admin-trash-2');
        }

        return array_merge($actions->toArray(), isset($sub_actions) ? $sub_actions->toArray() : []);
    }

    public function getSorted(Collection $collection)
    {
        $collection->load('page');
        $sort = function (Actu $actu) {
            return Arr::get(Arr::dot($actu->toArray()), request('orderBy', 'id'));
        };

        return [
            'actus' => request('order', 'asc') === "asc" ?
                $collection->sortBy($sort) :
                $collection->sortByDesc($sort),
            'order' => [
                'item'  => request('orderBy', 'id'),
                'order' => request('order', 'asc'),
            ]
        ];
    }

    public function index()
    {
        $actions = $this->getDefaultActions();

        // On remplace la 2ieme action du menu
        $actions['sub_actions'][0] = [
            'name' => 'Dépublier', 'href' => 'admin_actu_unpublish', 'picto' => 'bloomicon-admin-arrow-down-left'
        ];

        $data = array_merge($this->getSorted(Actu::published()->get()), $actions);

        return view('Actu::admin.listing', $this->getViewData($data));
    }

    public function programmed()
    {
        $data = array_merge($this->getSorted(Actu::programmed()->get()), $this->getDefaultActions());

        return view('Actu::admin.listing', $this->getViewData($data));
    }

    public function encours()
    {
        $data = array_merge($this->getSorted(Actu::status(2)->notProgrammed()->get()), $this->getDefaultActions());

        return view('Actu::admin.listing', $this->getViewData($data));
    }

    public function edit(Actu $actu)
    {
        $dossiers = Dossier::all();
        $categories = Categorie::notdeleted()->get();
        $page = $actu->page;
        if (request('dossier_id')) {
            if ($page === null) {
                $page = new Page();
            }
            $page->dossier_id = request('dossier_id');
        }

        return view('Actu::admin.edit', compact('actu', 'dossiers', 'page', 'categories'));
    }

    public function save(Actu $actu)
    {
        $user_id = Auth::user()->getAuthIdentifier();

        if (request('statut_id')) {
            $statut = request('statut_id');
        } else {
            if ($actu->page) {
                $statut = $actu->page->statut_id ?: Statut::EN_COURS;
            } else {
                $statut = Statut::EN_COURS;
            }
        }

        $actu->categorie_id = request('categorie_id') ? : null;

        if (!$actu->exists) {
            // Nouveau

            // On crée l'article
            $actu->contenu = request('contenu');
            $actu->save();

            // On crée la page
            $page = new Page();
            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->meta_titre = '';
            $page->meta_description = '';
            $page->created_by = $user_id;
            $page->last_modified_by = $user_id;
            $page->intro = request('intro');
            $page->dossier_id = request('dossier_id');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->multiplicity = Page::SINGLE;
            $page->image_opengraph = request('vignette');
            $page->mentions = request('mentions');

            if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
                $statut = 2;
            }
            if (request('statut_id') == 1) {
                $statut = 1;
                $page->to_publish_at = now();
            }

            $page->statut_id = $statut;
            $page->contenu()->associate($actu);
            $page->save();
        } else {
            // Edition

            /**
             * @var Page $page
             */
            $page = $actu->page;

            $actu->contenu = request('contenu');
            if ($actu->isDirty()) {
                // On mets à jour la page
                $page->last_modified_by = $user_id;
                $actu->save();
            }
            $page->titre = request('titre');
            $page->slug = request('slug');
            $page->intro = request('intro');
            $page->dossier_id = request('dossier_id');
            $page->to_publish_at = request('date_pub');
            $page->to_unpublish_at = request('date_depub');
            $page->mentions = request('mentions');


            if (request('date_pub') && Carbon::createFromFormat('Y-m-d', request('date_pub'))->isAfter(now())) {
                $statut = 2;
            }
            if (request('statut_id') == Statut::PUBLIE) {
                $statut = Statut::PUBLIE;
                $page->to_publish_at = now();
            }
            $page->statut_id = $statut;

            $page->image_opengraph = request('vignette');
            if ($page->isDirty()) {
                $page->last_modified_by = $user_id;
                $page->save();
            }
        }

        return redirect()->route('admin_actu_index');
    }

    public function preview(Actu $actu)
    {
        return $actu->getHandler()->single($actu);
    }

    public function publish(Actu $actu)
    {
        /**
         * @var Page $page
         */
        $page = $actu->page;
        $page->publish();

        return redirect()->back();
    }

    public function unpublish(Actu $actu)
    {
        /**
         * @var Page $page
         */
        $page = $actu->page;
        $page->unpublish();

        return redirect()->back();
    }

    public function delete(Actu $actu)
    {
        /**
         * @var Page $page
         */
        $page = $actu->page;
        $page->softDelete();

        return redirect()->back();
    }

    // Catégories

    public function categories()
    {
        $categories = Categorie::notdeleted()->orderBy('rank')->get();

        return view('Actu::admin.categories.listing', compact('categories'));
    }


    public function saveCategorie(Categorie $categorie)
    {
        $user_id = Auth::user()->getAuthIdentifier();

        $categorie->nom = request('nom');
        $categorie->couleur = request('couleur');


        if (!$categorie->exists) {
            // Nouveau

            // On crée la catétégorie
            $categorie->rank = Categorie::all()->count();
            $categorie->save();

            // On crée la page
            $page = new Page();
            $page->titre = request('nom');
            $page->meta_titre = '';
            $page->meta_description = '';
            $page->slug = request('nom');
            $page->created_by = $user_id;
            $page->last_modified_by = $user_id;
            $page->multiplicity = Page::SINGLE;

            if (request('actif')) {
                $page->statut_id = Statut::PUBLIE;
                $page->to_publish_at = now();
            }
            else {
                $page->statut_id = Statut::EN_COURS;
                $page->to_publish_at = null;
            }

            $page->contenu()->associate($categorie);
            $page->save();
        } else {
            // Edition

            /**
             * @var Page $page
             */
            $page = $categorie->page;
            $page->slug = request('nom');


            if ($categorie->isDirty()) {
                // On mets à jour la page
                $page->last_modified_by = $user_id;
                $categorie->save();
            }
            $page->titre = request('nom');

            if (request('actif')) {
                $page->statut_id = Statut::PUBLIE;
                $page->to_publish_at = now();
            }
            else {
                $page->statut_id = Statut::EN_COURS;
                $page->to_publish_at = null;
            }

            if ($page->isDirty()) {
                $page->last_modified_by = $user_id;
                $page->save();
            }
        }

        return redirect()->route('admin_actu_categories');
    }



    public function publishCategorie($id)
    {
        $categorie = Categorie::findOrFail($id);
        /**
         * @var Page $page
         */
        $page = $categorie->page;
        $page->publish();

        return (['success' => true]);
    }

    public function unpublishCategorie($id)
    {
        $categorie = Categorie::findOrFail($id);
        /**
         * @var Page $page
         */
        $page = $categorie->page;
        $page->unpublish();

        return (['success' => true]);
    }

    public function deleteCategorie(Categorie $categorie)
    {
        /**
         * @var Page $page
         */
        $page = $categorie->page;
        $page->softDelete();

        $actus = Actu::where('categorie_id', '=', $categorie->id)->get();

        foreach ($actus as $actu){
            /**
             * @var Actu $actu
             */
            $actu->page->archive();
            $actu->categorie_id = null;
            $actu->save();
        }

        return redirect()->back();
    }

    public function formCategorie(string $type, $id=null)
    {
        if($id){
            $categorie = Categorie::findOrFail($id);
        }
        $data = compact('categorie');

        return view('Actu::admin..categories.' . $type, $data);
    }

    public function reorderCategorie()
    {
        $ordre = request('ordre', []);

        $categories = Categorie::notdeleted()->get();

        foreach ($categories as $categorie) {
            /**
             * @var Categorie $categorie
             */
            $categorie->rank = array_search($categorie->id, $ordre, false);
            $categorie->save();
        }
    }
}
