<?php
/**
 * Created by IntelliJ IDEA.
 * User: Loïc Rameau
 * Date: 11/05/2016
 * Time: 15:48
 */

namespace Bloom\Framework;

use Assetic\Extension\Twig\AsseticExtension;
use Bloom\Configuration;
use Bloom\Controller\Request\Request;
use Bloom\Database\Database;
use Bloom\Framework\Controllers\BControllers;
use Bloom\Framework\Controllers\BModules;
use Bloom\Framework\View\BView;
use Bloom\Singleton;

class Application
{
    use Singleton;

    /**
     * @var \Twig_Environment
     */
    protected $twig;
    /**
     * @var string
     */
    protected $namespace = 'Bloom\Framework\\';

    /**
     * @var BModules[]
     */
    public static $modules = [];

    protected $regexP = '~^(?<mandatory>[^\[]+)?(\[(?<optional>.+)?\])?~';
    protected $regexP_named = '~{(?<name>\w+)(:(?<regx>[^}]+))?}~';

    /**
     * @var Configuration
     */
    public $configuration;
    /**
     */
    public $log;
    /**
     * @var Database
     */
    public $db;

    /**
     * @var \Twig_LoaderInterface
     */
    protected $loader;

    protected function __construct()
    {
        $this->loader = new \Twig_Loader_Filesystem(APP_PATH . '/views');
        $this->twig = new \Twig_Environment($this->loader, [
            'cache' => APP_ENV === 'prod' ? CACHE_PATH : false,
            'debug' => APP_ENV !== 'prod'
        ]);
        if (APP_ENV !== 'prod') {
            $this->twig->addExtension(new \Twig_Extension_Debug());
        }
        $this->configuration = Configuration::getInstance()->SetFile(CONF_PATH . '/' . APP_ENV)->HaveSection();
        $this->db = new Database(
            'mysql:host=' . Configuration::get('host', 'BDD') . ';dbname=' . Configuration::get('dbname', 'BDD'),
            Configuration::get('user', 'BDD'),
            Configuration::get('passwd', 'BDD'),
            [\PDO::MYSQL_ATTR_INIT_COMMAND => 'SET NAMES utf8']
        );
        $this->initSession();
        $this->genFilter();
        $this->genTest();
        $this->genFunction();
    }


    /**
     *
     */
    protected function genFunction()
    {
        $this->twig->addFunction(new \Twig_SimpleFunction('pop_errors', function () {
            $_SESSION[APP_NAME][APP_ENV]['error'] = [];
        }));
        $this->twig->addFunction(new \Twig_SimpleFunction('form_token', function ($salt = null) {
            if ($salt === null) {
                $salt = SID . '-' . date('Y-m-d H:i:s');
            }
            if (function_exists('password_hash')) {
                $hash = password_hash($salt, PASSWORD_BCRYPT);
            } else {
                $hash = md5(mt_rand(1, 1000000) . $salt);
            }
            $hash_ = Application::getSession('csrf_hash', []);
            $hash_[] = $hash;
            Application::setSession('csrf_hash', $hash_);
            return '<input type="hidden" name="token" value="' . $hash . '"/>';
        }, ['pre_escaped' => 'html', 'is_safe' => ['html']]));


        $this->twig->addFunction(new \Twig_SimpleFunction('modulesUrls', function ($name) {
            return Application::getModule($name)->getUrl();
        }));

        $this->twig->addFunction(new \Twig_SimpleFunction('getModule', function ($name) {
            return Application::getModule($name);
        }));

        $this->twig->addFunction(new \Twig_SimpleFunction('getUrl', function ($moduleName, $item) {
            return Application::makeUrlFromName($moduleName, $item);
        }));

        $this->twig->addFunction(new \Twig_SimpleFunction('moduleActif', function ($moduleName) {
            return Uri::getInstance()->match(Application::getModule($moduleName)->getUrl());
        }));
    }

    /**
     * @param string $moduleName
     * @return BModules|void
     */
    public static function getModule($moduleName)
    {
        foreach (Application::$modules as $module) {
            if ($module->getName() === $moduleName) {
                return $module;
            }
        }
        return;
    }

    /**
     * @param string $moduleName
     * @param string $item
     * @return string
     */
    public static function makeUrlFromName($moduleName, $item)
    {

        $module = static::getModule($moduleName);
        if($module) {
            return static::makeUrl($module, $item);
        }
        return '';
    }

    /**
     * @param Modules $module
     * @param mixed   $item
     * @return string
     */
    public static function makeUrl(Modules $module, $item)
    {
        return $module->makeUrl($item);
    }

    /**
     *
     */
    protected function genFilter()
    {
        $this->twig->addFilter(new \Twig_SimpleFilter('makeUri', function ($string, $abs = false) {
            if (preg_match('/^http:/', $string)) {
                return $string;
            }
            if (!preg_match('/^\//', $string)) {
                $string = '/' . $string;
            }
            switch (APP_ENV) {
                case 'prod':
                    return $abs ? 'http://' . $_SERVER['HTTP_HOST'] . $string : $string;
                    break;
                case 'preprod':
                    return $abs ? 'http://' . $_SERVER['HTTP_HOST'] . $string : $string;
                    break;
                case 'development' :
                default:
                    if (strpos(APP_PATH, '/app')) {
                        return $abs ? 'http://' . $_SERVER['HTTP_HOST'] . '/app' . $string : '/app' . $string;
                    } else {
                        return $abs ? 'http://' . $_SERVER['HTTP_HOST'] . '/dist' . $string : '/dist' . $string;
                    }
            }
        }));
        $this->twig->addFilter(new \Twig_SimpleFilter('ENV', function ($string) {
            return str_replace(BASE_PATH, '', constant($string));
        }));
    }

    /**
     *
     */
    protected function genTest()
    {
        $this->twig->addTest(new \Twig_SimpleTest('url_active', function ($url) {
            return preg_match('~^' . $url . '~', $_SERVER['REQUEST_URI']);
        }));
    }

    /**
     *
     */
    protected function setHeaders()
    {
        header('Content-Security-Policy: Sub-Resource Integrity (SRI)');
        header('X-XSS-Protection: 1');
        header('X-Content-Type-Options: nosniff');
    }

    /**
     * @return string
     */
    public function display()
    {
        $uri = Uri::getInstance();
        $this->setHeaders();

        if ($uri->getFragments(0) === 'logout') {
            unset($_SESSION);
            session_destroy();
            static::navigate('/home');
        }

        $tpl = '';
        if (array_key_exists('tpl', $_POST)) {
            $tpl = $_POST['tpl'];
        }


        foreach (static::$modules as $module) {
            if ($uri->match($module->getUrl())) {
                $module->setTpl($tpl ?: $uri->getGroup($module->getUrl()));
                return $module->execute();
            }
        }

        $controller_class = $this->namespace . 'Controller\\' . ucfirst($uri->getFragments(0)) . 'Controllers';
        if (class_exists($controller_class)) {
            /**
             * @var BControllers $controller
             */
            $controller = new $controller_class($this);
            $controller->setTpl($tpl ?: $uri->getGroup('/' . $uri->getFragments(0)));
            return $controller->execute();
        }
        $view_class = $this->namespace . 'View\\' . ucfirst($uri->getFragments(0)) . 'View';

        if (!class_exists($view_class)) {
            return static::render404();
        }
        /**
         * @var BView $view
         */
        $view = new $view_class();
        if (count($uri->getFragments()) > 1) {
            preg_match($this->regexP, $view_class::$parameter, $matches);
            $parameter = $this->parse_parameter($matches, $uri->getGroup('/' . $uri->getFragments(0)));
            $view->setParameter($parameter);
        }
        return $view->display($tpl);
    }

    public static function render404($msg = '404 Page Not Found')
    {
        foreach($_SESSION[APP_NAME][APP_ENV] as $key=>$value) {
            $datas[$key]=$value;
        }
        $datas = array_merge($datas, ['modules'=>Application::$modules]);
        $datas = array_merge($datas, ['user'=>Application::getSession('current_user')]);
        $datas = array_merge($datas, ['errors'=>$_SESSION[APP_NAME][APP_ENV]['error']]);
        $datas = array_merge($datas, ['error_msg'=> $msg] );
        return static::renderS('error.twig', $datas);
    }

    /**
     * @param string $template
     * @param array  $data
     * @return string
     */
    public static function renderS($template, array $data = [])
    {
        return static::getInstance()->render($template, $data);
    }

    /**
     * @param string $template
     * @param array  $data
     * @return string
     */
    public function render($template, array $data = [])
    {
        return $this->twig->render($template, $data);
    }

    /**
     * @param $namespace
     */
    public function setNameSpace($namespace)
    {
        $this->namespace = $namespace;
    }

    /**
     * @param array $matches
     * @param       $url
     * @return array
     */
    public function parse_parameter(array $matches, $url)
    {
        $mandatory = $matches['mandatory'];
        $optional = $matches['optional'];
        $parameter = [];
        preg_match($this->regexP_named, $mandatory, $matches_tmp);
        if ($matches_tmp) {
            if (!preg_match('~' . str_replace($matches_tmp, '(?<value>' . $matches_tmp['regx'] . ')', $mandatory) . '~', $url, $values)) {
                throw new \InvalidArgumentException('Invalid url exception');
            }
            $parameter[$matches_tmp['name']] = $values['value'];
        }
        preg_match($this->regexP_named, $optional, $matches_tmp);
        if ($matches_tmp) {
            if (!preg_match('~' . str_replace('[' . $matches_tmp[0] . ']', '(?<value>' . $matches_tmp['regx'] . ')', $optional) . '~', $url, $values)) {
                throw new \InvalidArgumentException('Invalid url exception');
            }
            $parameter[$matches_tmp['name']] = $values['value'];
        }
        return $parameter;
    }

    /**
     *
     */
    public function initSession()
    {
        // 2h de sessions
        // HttpOnly cookie
        session_set_cookie_params(2 * 3600, '/', '.' . $_SERVER['HTTP_HOST'], false, true);
        session_start();
        session_regenerate_id();
        if (!array_key_exists(APP_NAME, $_SESSION)) {
            $_SESSION[APP_NAME] = [
                APP_ENV => [
                    'session_start' => time(),
                    'error' => []
                ]
            ];
        }
    }

    /**
     * Get session variable
     * @param string $variable
     * @param null   $default
     * @return mixed|null
     */
    public static function getSession($variable, $default = null)
    {
        if (!array_key_exists($variable, $_SESSION[APP_NAME][APP_ENV])) {
            return $default;
        }

        return $_SESSION[APP_NAME][APP_ENV][$variable];
    }

    /**
     * Set a session variable
     * @param string $variable
     * @param mixed  $value
     */
    public static function setSession($variable, $value)
    {
        $_SESSION[APP_NAME][APP_ENV][$variable] = $value;
    }

    /**
     * @param string $url
     */
    public static function navigate($url)
    {
        header('Location: ' . $url);
        exit(0);
    }

    /**
     * @param Modules $controller
     */
    public function registerModules(Modules $controller)
    {
        if ($controller->getFactory() !== null) {
            $this->twig->addExtension(new AsseticExtension($controller->getFactory()));
        }
        $path = $controller->getViewPath();
        if ($path !== '') {
            if (is_array($path)) {
                foreach ($path as $key => $value) {
                    if (is_numeric($key)) {
                        $this->loader->addPath($value);
                    } else {
                        $this->loader->addPath($value, $key);
                    }
                }
            } else {
                $this->loader->addPath($path);
            }
        }
        static::$modules[] = $controller;
    }

    /**
     * @param $message
     * @param $detail
     */
    public static function reportError($message, $detail)
    {
        $_SESSION[APP_NAME][APP_ENV]['error'][] = ['message' => $message, 'detail' => $detail];
    }

    /**
     * @return bool
     */
    public static function validFormHash()
    {
        $token = Request::getInstance()->get('token', false);
        $error = error_reporting(-1);
        unset($_POST['token'], $_REQUEST['token'], $_GET['token']);
        error_reporting($error);
        $ok = in_array($token, $hash_ = static::getSession('csrf_hash', []), true);
        if ($ok) {
            unset($hash_[array_search($token, $hash_, true)]);
            static::setSession('csrf_hash', $hash_);
        }
        return $ok;
    }

    /**
     * @return bool
     */
    public static function isCli()
    {
        return php_sapi_name() === 'cli';
    }

    /**
     * @return \Twig_Environment
     */
    public function getTwig()
    {
        return $this->twig;
    }

    /**
     * @return \Twig_Loader_Filesystem|\Twig_LoaderInterface
     */
    public function getLoader()
    {
        return $this->loader;
    }

    /**
     * @return bool
     */
    public static function isPost()
    {
        return Uri::getInstance()->isPost();
    }

    /**
     * @return bool
     */
    public static function isGet()
    {
        return Uri::getInstance()->isGet();
    }

    /**
     * @return bool
     */
    public static function isPut()
    {
        return Uri::getInstance()->isPut();
    }

    /**
     * @return bool
     */
    public static function isDelete()
    {
        return Uri::getInstance()->isDelete();
    }
}
