<?php
/**
 * Created by IntelliJ IDEA.
 * User: Loïc Rameau
 * Date: 24/10/2016
 * Time: 16:44
 */

namespace Bloom\Framework;

/**
 * Class Uri
 * @package Bloom\Framework
 */
class Uri
{
    /**
     * Called uri
     * @var string
     */
    protected $uri;
    /**
     * Fragments list, /fragements0/fragments1/...
     * @var array
     */
    protected $fragments;
    /**
     * Environment parameter
     * @var array
     */
    protected $parameter;
    /**
     * Current instance of uri
     * @var Uri
     */
    static protected $instance;

    /**
     * Uri constructor.
     * @param string $uri
     * @param array  $fragments
     * @param array  $parameter
     */
    protected function __construct($uri = '', array $fragments = [], array $parameter = [])
    {
        $this->uri = $uri;
        $this->fragments = $fragments;
        $this->parameter = $parameter;
    }

    /**
     * Create Uri from environment parameters, such as $_SERVER
     * @param array $environment
     * @return Uri
     */
    public static function createFromEnvironment(array $environment)
    {
        $home = dirname($environment['SCRIPT_FILENAME']);
        $home_dir = str_replace($environment['CONTEXT_DOCUMENT_ROOT'], '', $home);
        $uri = '/';
        if(array_key_exists('REDIRECT_URL', $environment)) {
            $uri = str_replace($home_dir, '', $environment['REDIRECT_URL']);
        }
        $environment['doc_root'] = $home;
        $environment['baseDir'] = $home_dir;
        if (array_key_exists('view', $_REQUEST) && $uri !== $_REQUEST['view']) {
            $uri = '/' . $_REQUEST['view'];
        }
        $fragments = explode('/', $uri);
        unset($fragments[0]);
        return static::$instance = new static($uri, array_values($fragments), $environment);
    }

    /**
     * Flush and get instance
     * @param array $environment
     * @return Uri
     */
    public static function create(array $environment = null)
    {
        return static::createFromEnvironment($environment?:$_SERVER);
    }

    /**
     * Get current uri instance
     * @return Uri
     */
    public static function getInstance()
    {
        if (static::$instance === null) {
            return static::createFromEnvironment($_SERVER);
        }
        return static::$instance;
    }

    /**
     * Get a fragments or all the fragments
     * @param int $index
     * @return string|array
     */
    public function getFragments($index = null)
    {
        if ($index === null) {
            return $this->fragments;
        }
        return $this->fragments[$index];
    }

    /**
     * Get uri
     * @return string
     */
    public function getUri()
    {
        return $this->uri;
    }

    /**
     * Get a parameter, value in $environment
     * @param $parameter
     * @return null
     */
    public function get($parameter)
    {
        if(array_key_exists($parameter, $this->parameter)) {
            return $this->parameter[$parameter];
        }
        return null;
    }

    /**
     * Does the uri match the current one
     * @param $uri
     * @return int
     */
    public function match($uri)
    {
        return preg_match('~^' . $uri . '~', $this->uri);
    }

    /**
     * Does the current uri match the given group
     * Ex : /foo/bar match the group /foo
     * @param string $uri
     * @return string The matching result, /foo/bar return /bar
     */
    public function getGroup($uri)
    {
        if (preg_match('~^' . $uri . '(/.+)~', $this->uri, $matches)) {
            return $matches[1];
        }
        return '';
    }

    /**
     * Does the request is POST
     * @return bool
     */
    public function isPost()
    {
        return $this->parameter['REQUEST_METHOD'] === 'POST';
    }

    /**
     * Does the request is GET
     * @return bool
     */
    public function isGet()
    {
        return $this->parameter['REQUEST_METHOD'] === 'GET';
    }

    /**
     * Does the request is PUT
     * @return bool
     */
    public function isPut()
    {
        return $this->parameter['REQUEST_METHOD'] === 'PUT';
    }

    /**
     * Does the request is DELETE
     * @return bool
     */
    public function isDelete()
    {
        return $this->parameter['REQUEST_METHOD'] === 'DELETE';
    }

    /**
     * Get root directory
     * @return mixed
     */
    public function getHomeDir()
    {
        return $this->parameter['doc_root'];
    }

    /**
     * Get relative directory
     * @return mixed
     */
    public function getRelatifPath() {
        return $this->parameter['baseDir'];
    }
}
